<?php

namespace Mainto\MRPCTool\Command;

use Illuminate\Console\Command;
use Illuminate\Support\Str;
use Throwable;

class RpcServiceApiDump extends Command {
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'rpc:dump-service-api {--dump_path=} {--base_namespace=} {--set_version=}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'RPC Service Api Dumper';

    public ?string $dumpPath;
    public ?string $baseNamespace;

    /**
     * Execute the console command.
     *
     * @return mixed
     * @throws Throwable
     */
    public function handle () {
        $dump_path = $this->option('dump_path');
        $base_namespace = $this->option('base_namespace');
        $this->dumpPath = rtrim($dump_path ?: format_path('.'), '/').'/';
        $this->baseNamespace = $base_namespace ?: 'App\Rpc\Controllers\\';
        $microName = studly_case(config('rpc-server.service_name', 'NoName'));

        $service = [
            'serviceName' => $microName,
            'name'        => Str::kebab($microName),
            'version'     => $this->getVersion(),
            'classes'     => [],
        ];

        foreach (RpcRouterExtend::getRouterClassStruct($this->baseNamespace, true, false, true) as $className => $item) {
            $item['namespace'] = $item['namespace'] ? $microName.'\\'.$item['namespace'] : $microName;
            $methods = $item['methods'];
            foreach ($methods as &$method) {
                $method['params'] = array_values($method['params']);
                $method = array_only($method, ["alias", "name", "isDeprecated", "params", "url", "method", "returnType",]);
            }
            $service['classes'][] = [
                'methods'       => $methods,
                'className'     => $className,
                'classShotName' => $item['shortName'],
            ];
        }

        file_exists($this->dumpPath) || mkdir($this->dumpPath, 0777, true);
        $serviceJsonPath = $this->dumpPath."service.json";
        file_put_contents($serviceJsonPath, json_encode($service, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));
        $this->comment(">>>> write service.json {serviceName: {$microName}, version: {$this->getVersion()} ...} to {$serviceJsonPath}");

        return 0;
    }

    public function getVersion () {
        $version = $this->option('set_version');

        if (!$version) {
            $version = get_git_branch_version();
        }
        return $this->parseStr($version);
    }

    private function parseStr ($str) {
        return ltrim(preg_replace('/[^\w]/m', '.', $str), 'release.');
    }
}
