<?php

namespace Mainto\MRPCTool\Command;

use Illuminate\Console\Command;
use Illuminate\Support\Str;
use Mainto\MRPC\Server\RpcRouter;
use ReflectionClass;
use ReflectionException;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputOption;
use Throwable;

class RpcMakeControllerTestClass extends Command {

    protected function configure () {
        parent::configure();
        $this->setName('rpc:make-controller-test')
            ->setDescription('Create a new controller test class ')
            ->addArgument('class_name', InputArgument::REQUIRED, 'The registered controller name')
            ->addOption('controller_namespace_base', 'c', InputOption::VALUE_OPTIONAL, 'The controller namespace base', 'App\Rpc\Controllers')
            ->addOption('test_class_namespace_base', 't', InputOption::VALUE_OPTIONAL, 'The test class namespace base', 'Tests\Feature\Controllers')
            ->addOption('output_dir', 'o', InputOption::VALUE_OPTIONAL, 'The test class output dir', './tests/Feature/Controllers/');
    }

    /**
     * ===========================================================
     * Dump IDE End
     */

    public string $testNamespaceBase;
    public string $controllerNamespace;

    public string $testPathBase;

    /**
     * Execute the console command.
     *
     * @return mixed
     * @throws ReflectionException
     * @throws Throwable
     */
    public function handle () {
        $className = $this->argument('class_name');
        $this->testNamespaceBase = trim($this->option('test_class_namespace_base'));
        $this->controllerNamespace = strval($this->option('controller_namespace_base'));
        $this->testPathBase = format_path($this->option('output_dir'));
        $classInstancesMap = RpcRouter::getInstance()->getClassInstancesMap();

        if ($className === '*') {
            $classes = $classInstancesMap;
        } else {
            if (!isset($classInstancesMap[$className])) {
                $this->error("The class is not defined in rpc route! ");
                exit;
            }
            $classes = [$classInstancesMap[$className]];
        }

        foreach ($classes as $classItem) {
            ["classInstance" => $class, "methodItems" => $methodItems] = $classItem;
            $namespace = explode('\\', str_replace($this->controllerNamespace, '', get_class($class)));
            $unset_controller_name = array_slice($namespace, 0, count($namespace) - 1);
            $class_name = array_pop($namespace);
            $test_namespace = implode('\\', $unset_controller_name);
            $dir_name = $this->testPathBase.implode('/', $unset_controller_name);
            $file_name = $dir_name.'/'.$class_name.'Test.php';
            !file_exists($dir_name) and mkdir($dir_name, 0777, true);
            $methods = array_keys($methodItems);
            if (!file_exists($file_name)) {
                file_put_contents($file_name, "<?php\n".view('test-controller', [
                        'controllerDocNamespace'  => config('rpc-tool.cmd.namespace.controller_doc'),
                        'class_name'              => $class_name,
                        'controllerNamespaceBase' => $this->controllerNamespace.($test_namespace ? $test_namespace."\\".$class_name : '\\'.$class_name),
                        'methods'                 => $methods,
                        'namespace'               => $this->testNamespaceBase.($test_namespace ? $test_namespace : ''),
                        'package'                 => '@package '.$this->testNamespaceBase.($test_namespace ? $test_namespace : ''),
                    ])->render());
            } else {
                $this->rebuildTestClassDoc($file_name, $class_name, $methods);
            }
        }

        return 0;
    }

    /**
     * @param $file
     * @param $class_name
     * @param $methods
     * @throws ReflectionException
     */
    public function rebuildTestClassDoc ($file, $class_name, $methods) {
        $testClass = $this->testNamespaceBase.str_replace(
                ['.php', realpath($this->testPathBase), '/'],
                ['', '', '\\'],
                realpath($file)
            );
        $refClass = new ReflectionClass($testClass);
        $package = substr($refClass->name, 0, -(strlen($refClass->getShortName())) - 1);

        $doc = <<<EOF
/**
 * Class {$class_name}Test
 * @package {$package}
 *
 * --------------------------------------
 * auto gen method annotations, if the method name is already exists,
 * please use getResponse([method_name], [params])
 * --------------------------------------
EOF;
        foreach ($methods as $method) {
            $method = Str::ucfirst($method);
            $doc .= "\n * @method call$method(array \$params = [], bool \$enable_doc = true)";
        }
        $doc .= "\n */";

        file_put_contents($file, str_replace($refClass->getDocComment(), $doc, file_get_contents($file)));
    }
}