<?php
/**
 * Created by PhpStorm.
 * User: liuzaisen
 * Date: 2018/11/4
 * Time: 11:19 PM
 */

namespace Mainto\RpcTool\TestHelper;


use Closure;
use Doctrine\Common\Annotations\AnnotationException;
use Doctrine\Common\Annotations\AnnotationReader;
use Illuminate\Validation\ValidationException;
use Mainto\RpcServer\RpcAnnotations\Alias;
use ReflectionException;
use ReflectionMethod;
use Throwable;

trait BuildDoc {
    private string $calledClass;
    private string $calledMethod;

    /**
     * @param Closure $execute
     * @param string $calledClass
     * @param string $calledMethod
     * @param array $params
     * @return mixed
     * @throws ValidationException
     * @throws AnnotationException
     * @throws ReflectionException
     * @throws Throwable
     */
    public function dump (Closure $execute, string $calledClass, string $calledMethod, $params = []) {
        $this->calledClass = $calledClass;
        $this->calledMethod = $calledMethod;
        try {
            $response = $execute();
            if (self::canDump()) {
                //build request doc
                $this->buildDoc($calledMethod, $params, null, $response);
            }

        } catch (Throwable $e) {
            if (self::canDump()) {
                //build request doc
                $this->buildDoc($calledMethod, $params, $e, null);
            }
            if ($e instanceof ValidationException) {
                dump($e->errors());
            }
            throw $e;
        }
        return $response;
    }

    /**
     * @param $method
     * @param $params
     * @param $e
     * @param $response_body
     * @throws AnnotationException
     * @throws ReflectionException
     */
    private function buildDoc ($method, $params, $e, $response_body): void {
        $document_file_path = config('rpc-tool.micro_doc.doc_dir').'/micro_document.json';
        if (file_exists($document_file_path)) {
            $document = unserialize(file_get_contents($document_file_path));
        } else {
            $document = Document\Document::getInstance();
        }
        $group = $document->getGroup($this->calledClass);
        $api = $group->findGroupApi($this->calledClass, $method);
        $refTestMethod = $this->getCalledTestMethod(get_called_class(), debug_backtrace(false, 10));

        [$title, $description] = $this->getTestMethodTitleAndDescription($refTestMethod);

        /** @var Throwable $e */
        if ($e) {
            $response_body = [
                'error_code' => $e->getCode(),
                'success'    => false,
                'error_msg'  => $e instanceof ValidationException ? $e->errors() : $e->getMessage(),
            ];
        } else {
            $response_body = [
                'success' => true,
                'msg'     => $response_body,
            ];
        }
        $api->addRequestExample($title, $params, $response_body, $description);
        file_put_contents($document_file_path, serialize($document));

    }

    /**
     * 获取对应测试的方法注释  必须包含 Alias 用于请求示例的标题
     *
     * @param $refTestMethod
     * @return array
     */
    private function getTestMethodTitleAndDescription (ReflectionMethod $refTestMethod) {
        $reader = new AnnotationReader();
        $annotations = $reader->getMethodAnnotations($refTestMethod);
        $test_title = '';
        foreach ($annotations as $annotation) {
            if ($annotation instanceof Alias) {
                $test_title = $annotation->name;
                break;
            }
        }
        if (!$test_title) {
            $test_title = $refTestMethod->getDeclaringClass()->getShortName()."::".$refTestMethod->name;
        }
        $test_description = '';
        $docComment = $refTestMethod->getDocComment();
        foreach (explode("\n", $docComment) as $line) {
            if (strpos($line, '@') === false && strpos($line, '/') === false) {
                $_doc = trim(str_replace('*', '', $line));
                if (!$_doc) continue;
                $test_description .= $_doc."\n";
                continue;
            }
        }

        return [$test_title, trim($test_description)];
    }

    /**
     * 获取调用 getResponse 的对应测试function 的ReflectionMethod实例
     *
     * @param $called_class
     * @param $debug_backtrace
     * @return ReflectionMethod
     * @throws ReflectionException
     */
    private function getCalledTestMethod ($called_class, $debug_backtrace) {
        $begin = false;
        $called_method_name = '';
        foreach ($debug_backtrace as $debug) {
            if (!$begin and $debug['class'] === $called_class
                && ($debug['function'] == "__call" || $debug['function'] == "callAndDoc")) {
                $begin = true;
                continue;
            }
            if ($debug['class'] === 'PHPUnit\Framework\TestCase') break;
            if ($begin) {
                $called_method_name = $debug['function'];
            }
        }
        return new ReflectionMethod($called_class, $called_method_name);
    }

    public static function canDump () {
        return config('rpc-tool.micro_doc.enable');
    }

}