<?php
namespace Mainto\Mongodb;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Jenssegers\Mongodb\Collection as MongoDBCollection;
use Jenssegers\Mongodb\Connection;
use Jenssegers\Mongodb\Query\Builder;
use Mainto\Mongodb\Traits\MongoTransTrait;
use MongoDB\BSON\ObjectId;
use MongoDB\BSON\UTCDateTime;

class Mongodb {
    use MongoTransTrait;

    /**
     * @var Connection
     */
    private $db;

    /**
     * PoolRepository constructor.
     */
    public function __construct () {
        $this->db = DB::connection('mongodb');
    }

    /**
     * 插入一条
     *
     * @param string $name
     * @param int $id
     * @param array $item
     * @return string
     */
    public function addItem (string $name, array $item): string {
        return $this->getBuilder($name)->insertGetId($item);
    }

    /**
     * 获取查询 Builder
     *
     * @param string $name
     * @return Builder
     */
    public function getBuilder (string $name): Builder {
        return $this->db->collection($name);
    }

    /**
     * 插入多条
     *
     * @param string $name
     * @param array $items [$id => $item]
     * @return bool
     */
    public function addItems (string $name, array $items): bool {
        return $this->getBuilder($name)->insert(
            collect($items)
                ->map(function ($item, $id) {
                    $item['businessId'] = $id;

                    return $item;
                })
                ->values()
                ->all()
        );
    }

    /**
     * 更新信息
     *
     * @param string $name
     * @param string $objectId
     * @param array $item
     * @return bool
     */
    public function updateItem (string $name, string $objectId, array $item): bool {
        /** @var Builder $query */
        $query = $this->getBuilder($name)->where('_id', new ObjectId($objectId));
        return boolval($query->update($item, ['upsert' => true]));
    }

    /**
     * 更新信息 by cond
     *
     * @param string $name
     * @param array $cond
     * @param array $item
     * @return bool
     */
    public function updateItemByCond (string $name, array $cond, array $item): bool {
        /** @var Builder $query */
        $query = $this->getBuilder($name)->where($cond);
        return boolval($query->update($item, ['upsert' => true]));
    }

    /**
     * 移除项目
     *
     * @param string $name
     * @param array $ids
     * @return int
     */
    public function deleteItemsByIds (string $name, array $ids): int {
        return intval(
            $this->getBuilder($name)
                ->where([
                    '_id' => [
                        '$in' => collect($ids)->map(function ($id) {
                            return new ObjectId($id);
                        })->all(),
                    ],
                ])
                ->delete()
        );
    }

    /**
     * 移除项目 by cond
     *
     * @param string $name
     * @param array $cond
     * @return int
     */
    public function deleteItemsByCond (string $name, array $cond): int {
        return intval(
            $this->getBuilder($name)
                ->where($cond)
                ->delete()
        );
    }

    /**
     * 根据条件 查找一条
     *
     * @param string $name
     * @param array $cond
     * @return Model|\Illuminate\Database\Query\Builder|mixed|object|null
     */
    public function findItemByCond (string $name, array $cond) {
        $this->transformCond($cond);
        return $this->castMongoDocumentArrayToArray(
            $this->getBuilder($name)->where($cond)->first()
        );
    }

    private function transformCond (&$cond) {
        // 对_id 做单独转换
        if (isset($cond['_id']) && isset($cond['_id']['$in'])){
            foreach ($cond['_id']['$in'] as &$_id){
                $_id = new ObjectId($_id);
            }
        }
    }

    /**
     * 根据 ID 查找一条
     * @param string $name
     * @param string $id
     * @return array
     */
    public function findItemByObjectId (string $name, string $id): array {
        return $this->castMongoDocumentArrayToArray(
            $this->db->collection($name)->where('_id', new ObjectId($id))->first()
        );
    }

    /**
     * 搜索
     *
     * @param string $name
     * @param UTCDateTime $startAt
     * @param UTCDateTime $endAt
     * @param array $cond
     * @param string $direction
     * @param int $page
     * @param int $pageSize
     * @return array
     */
    public function search (string $name, array $cond, array $sort, int $page, int $pageSize = 10): array {
        $this->transformCond($cond);
        $query = $this->db->collection($name)
            ->where($cond);
        if ($sort) {
            $query->orderBy(array_key_first($sort), Arr::first($sort));
        }

        $ret = $query->paginate($pageSize, ['*'], 'page', $page);

        return [
            'data'      => $this->castMultiToArray($ret->items()),
            'last_page' => $ret->lastPage(),
            'total'     => $ret->total(),
        ];
    }

    /**
     * 聚合
     *
     * @param string $name
     * @param array $cond
     * @param array $aggregateRule
     * @param bool $withArray
     * @return array
     */
    public function aggregate (string $name, array $cond, array $aggregateRule, bool $withArray = false): array {
        $ret = $this->db->collection($name)->raw(function (MongoDBCollection $collection) use ($cond, $aggregateRule) {
            if ($cond) array_unshift($aggregateRule, ['$match' => $cond]);
            return $collection->aggregate($aggregateRule);
        });
        $res = $ret->toArray();
        if (count($res) > 1 || $withArray) {
            return ['result' => $this->castMultiToArray($res)];
        } else {
            if (!$res) return [];
            return $this->castMongoDocumentToArray(Arr::first($res));
        }
    }
}
