<?php
namespace Mainto\Bridge\Invokes\User\Rpc;

use Mainto\RpcServer\RpcClient\RpcClass;
use Mainto\RpcServer\RpcClient\RpcMockClass;
use Mainto\RpcServer\RpcServer\RpcDefinition;
use InvalidArgumentException;

/**
 * Class Unique
 * @method addUserByParams(array|\Mainto\Bridge\Structs\User\Rpc\Cmd\User\AddUserCmd $params)
 * @method editUserByParams(array|\Mainto\Bridge\Structs\User\Rpc\Cmd\User\EditUserCmd $params)
 * @method checkUserPasswordByParams(array $params)
 * @method changeUserPasswordByParams(array $params)
 * @method forgetUserPasswordByParams(array $params)
 * @method logoutAndChangeUserPhoneByParams(array $params)
 * @method getUserBaseInfoByPhoneByParams(array $params)
 * @method getUserInfoByUserIdByParams(array $params)
 * @method getUserInfoByPhoneOrUserIdByParams(array $params)
 * @method getUsersByParams(array $params)
 * @method getUserListByParams(array|\Mainto\Bridge\Structs\User\Rpc\Cmd\User\UserListCmd $params)
 * @method deleteUserByParams(array $params)
 */
class User {
    private static $instance;

    /**
     * 获取一个远程调用类
     * @return  static
     */
    public static function getDirectClass () {
        if (self::$instance == null) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    /**
     * 新增用户
     *
     * @var  \Mainto\Bridge\Structs\User\Rpc\Cmd\User\AddUserCmd $addUserCmd
     * @throws  \Exception
     * @return  \Mainto\Bridge\Structs\User\Rpc\DTO\User\UserBaseDTO
     */
    public static function addUser (\Mainto\Bridge\Structs\User\Rpc\Cmd\User\AddUserCmd $addUserCmd) {
        return app('json_mapper')->mapUseTypeString(
            self::getClass()->addUser($addUserCmd),
            "\Mainto\Bridge\Structs\User\Rpc\DTO\User\UserBaseDTO"
        );
    }


    /**
     * 编辑用户
     *
     * @var  \Mainto\Bridge\Structs\User\Rpc\Cmd\User\EditUserCmd $editUserCmd
     * @throws  \Exception
     * @return  \Mainto\Bridge\Structs\User\Rpc\DTO\User\UserBaseDTO
     */
    public static function editUser (\Mainto\Bridge\Structs\User\Rpc\Cmd\User\EditUserCmd $editUserCmd) {
        return app('json_mapper')->mapUseTypeString(
            self::getClass()->editUser($editUserCmd),
            "\Mainto\Bridge\Structs\User\Rpc\DTO\User\UserBaseDTO"
        );
    }


    /**
     * 核对用户密码
     *
     * @param  string $phone 用户手机号 [ require true ]
     * @param  string $password 用户密码 [ require true ]
     * @throws  \Exception
     * @return  bool
     */
    public static function checkUserPassword (string $phone, string $password) {
        return self::getClass()->checkUserPassword([
            "phone" => $phone,
            "password" => $password,
        ]);
    }

    /**
     * 用户修改密码
     *
     * @param  int $userId 用户id [ require true ]
     * @param  string $oldPassword 用户老密码, 可以为空，设置新密码 [ require false ]
     * @param  string $newPassword 用户新密码 [ require true ]
     * @throws  \Exception
     * @return  bool
     */
    public static function changeUserPassword (int $userId, string $newPassword, string $oldPassword = '') {
        return self::getClass()->changeUserPassword([
            "userId" => $userId,
            "newPassword" => $newPassword,
            "oldPassword" => $oldPassword,
        ]);
    }

    /**
     * 用户忘记密码
     *
     * @param  string $phone 用户手机号 [ require true ]
     * @param  string $captcha 验证码 [ require true ]
     * @param  string $password 密码 [ require true ]
     * @throws  \Exception
     * @return  bool
     */
    public static function forgetUserPassword (string $phone, string $captcha, string $password) {
        return self::getClass()->forgetUserPassword([
            "phone" => $phone,
            "captcha" => $captcha,
            "password" => $password,
        ]);
    }

    /**
     * 注销并修改用户手机号
     *
     * @param  string $newPhone 新手机号 [ require true ]
     * @param  string $oldPhone 旧手机号 [ require true ]
     * @param  string $code 验证码 [ require true ]
     * @param  string $certificate 用户沟通凭证 [ require true ]
     * @throws  \Exception
     * @return  bool
     */
    public static function logoutAndChangeUserPhone (string $newPhone, string $oldPhone, string $code, string $certificate) {
        return self::getClass()->logoutAndChangeUserPhone([
            "newPhone" => $newPhone,
            "oldPhone" => $oldPhone,
            "code" => $code,
            "certificate" => $certificate,
        ]);
    }

    /**
     * 通过手机号查询用户基础信息, 不存在可以先添加再返回
     *
     * @param  string $phone 用户手机号 [ require true ]
     * @param  bool $noExistAddNew 如果不存在是否新增一个 [ require false ]
     * @throws  \Exception
     * @return  \Mainto\Bridge\Structs\User\Rpc\DTO\User\UserBaseDTO
     */
    public static function getUserBaseInfoByPhone (string $phone, bool $noExistAddNew = false) {
        return app('json_mapper')->mapUseTypeString(
            self::getClass()->getUserBaseInfoByPhone([
                            "phone" => $phone,
                            "noExistAddNew" => $noExistAddNew,
                        ]),
            "\Mainto\Bridge\Structs\User\Rpc\DTO\User\UserBaseDTO"
        );
    }

    /**
     * 查询基础用户信息
     *
     * @param  int $userId 用户id [ require true ]
     * @param  bool $withTags 是否包含标签 [ require false ]
     * @throws  \Exception
     * @return  \Mainto\Bridge\Structs\User\Rpc\DTO\User\UserBaseWithTagsDTO
     */
    public static function getUserInfoByUserId (int $userId, bool $withTags = false) {
        return app('json_mapper')->mapUseTypeString(
            self::getClass()->getUserInfoByUserId([
                            "userId" => $userId,
                            "withTags" => $withTags,
                        ]),
            "\Mainto\Bridge\Structs\User\Rpc\DTO\User\UserBaseWithTagsDTO"
        );
    }

    /**
     * 查询用户信息, 包括地址，注册和tag信息
     *
     * @param  string $phone 用户手机号 [ require true ]
     * @param  int $userId 用户id [ require true ]
     * @throws  \Exception
     * @return  \Mainto\Bridge\Structs\User\Rpc\DTO\User\UserBaseWithDTO
     */
    public static function getUserInfoByPhoneOrUserId (string $phone, int $userId) {
        return app('json_mapper')->mapUseTypeString(
            self::getClass()->getUserInfoByPhoneOrUserId([
                            "phone" => $phone,
                            "userId" => $userId,
                        ]),
            "\Mainto\Bridge\Structs\User\Rpc\DTO\User\UserBaseWithDTO"
        );
    }

    /**
     * 获取基础用户数组数据
     *
     * @param  array $userIds 用户ids [ require true ]
     * @param  bool $withTags 是否包含标签 [ require false ]
     * @throws  \Exception
     * @return  \Mainto\Bridge\Structs\User\Rpc\DTO\User\UserBaseWithTagsDTO[]
     */
    public static function getUsers (array $userIds, bool $withTags = false) {
        return app('json_mapper')->mapUseTypeString(
            self::getClass()->getUsers([
                            "userIds" => $userIds,
                            "withTags" => $withTags,
                        ]),
            "\Mainto\Bridge\Structs\User\Rpc\DTO\User\UserBaseWithTagsDTO[]"
        );
    }

    /**
     * 获取用户列表
     *
     * @var  \Mainto\Bridge\Structs\User\Rpc\Cmd\User\UserListCmd $userListCmd
     * @throws  \Exception
     * @return  \Mainto\Bridge\Structs\User\Rpc\DTO\User\UserListDTO
     */
    public static function getUserList (\Mainto\Bridge\Structs\User\Rpc\Cmd\User\UserListCmd $userListCmd) {
        return app('json_mapper')->mapUseTypeString(
            self::getClass()->getUserList($userListCmd),
            "\Mainto\Bridge\Structs\User\Rpc\DTO\User\UserListDTO"
        );
    }


    /**
     * 删除用户
     *
     * @param  int $userId 用户id [ require true ]
     * @throws  \Exception
     * @return  bool
     */
    public static function deleteUser (int $userId) {
        return self::getClass()->deleteUser([
            "userId" => $userId,
        ]);
    }

    /**
     * getClass
     * @deprecated
     * @throws  \Exception
     * @return  mixed
     */
    public static function getClass () {
        return RpcClass::getClass('User', 'Rpc\User');
    }

    /**
     * registerMock
     * @throws  \Exception
     * @return  RpcMockClass
     */
    public static function registerMock () {
        return RpcClass::registerMockClass('User', 'Rpc\User');
    }

    /**
     * @param  $name
     * @param  $arguments
     * @return  mixed
     * @throws  \Exception
     */
    public function __call ($name, $arguments) {
        if (substr($name, -8, 8) === 'ByParams') {
            $shortName = str_replace('ByParams', '', $name);
            return self::getClass()->$shortName(...$arguments);
        }

        throw new InvalidArgumentException('Method not found! : '.$name);
    }
}