<?php
namespace Mainto\Bridge\Invokes\Store;

use Mainto\RpcServer\RpcClient\RpcClass;
use Mainto\RpcServer\RpcClient\RpcMockClass;
use InvalidArgumentException;

/**
 * Class Unique
 * @method getHaveStoreCityByParams(array $params)
 * @method searchStoreByDataByParams(array $params)
 * @method getStoreAppointableTimeByParams(array $params)
 * @method getStoreInfoByIdsByParams(array $params)
 * @method getStoreOpenSortByParams(array $params)
 * @method getStoreInfoByIdByParams(array $params)
 * @method getStoresByParams(array $params)
 * @method getStoresWithoutPageByParams(array $params)
 * @method getStoreInfoByNameByParams(array $params)
 * @method getMaxStoreIdInAreaByParams(array $params)
 * @method createStoreByParams(array $params)
 * @method editStoreByParams(array $params)
 * @method editStoreStartBusinessByParams(array $params)
 * @method editStoreServiceTimesByParams(array $params)
 * @method editStoreServiceIntervalByParams(array $params)
 * @method deleteStoreByParams(array $params)
 * @method restoreStoreByParams(array $params)
 * @method getStoreIdByDianpingIdByParams(array $params)
 * @method getDianpingIdByStoreIdByParams(array $params)
 */
class Store {
    private static $instance;

    /**
     * 获取一个远程调用类
     * @return  static
     */
    public static function getDirectClass () {
        if (self::$instance == null) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    /**
     * 获得当前有门店的所有城市
     *
     * @param  array $store_type 门店类型，不传则不指定 [ require true ]
     * @param  bool $includeDelete 包含删除的店铺 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function getHaveStoreCity (array $store_type = [], bool $includeDelete = false) {
        return self::getClass()->getHaveStoreCity([
            "store_type" => $store_type,
            "includeDelete" => $includeDelete,
        ]);
    }

    /**
     * 获取指定条件下的所有门店
     *
     * @param  array $city_ids 城市的ID数组 [Original type array&lt;uint&gt;] [ require true ]
     * @param  array $store_type 限定的门店类型 [Original type array&lt;string&gt;] [ require true ]
     * @param  string $type 获取类型 [ require true ]
     * @param  bool $includeDelete 包含删除的店铺 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function searchStoreByData (array $city_ids = [], array $store_type = [], string $type = 'base', bool $includeDelete = false) {
        return self::getClass()->searchStoreByData([
            "city_ids" => $city_ids,
            "store_type" => $store_type,
            "type" => $type,
            "includeDelete" => $includeDelete,
        ]);
    }

    /**
     * 此函数获取的仅仅是门店配置上可预约的时间点, 和排单表无关
     *
     * @param  int $store_id 门店ID [Original type uint] [ require true ]
     * @param  int $next_num 需要的时间点数量 [Original type uint] [ require true ]
     * @param  string $start_at 需要从什么时间开始 [ require true ]
     * @support  string $start_time 仅start_at为custom时有效，开始的时间点 
     * @param  bool $return_all 是否需要全部内容 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function getStoreAppointableTime (int $store_id, int $next_num, string $start_at = 'day', bool $return_all = false) {
        return self::getClass()->getStoreAppointableTime([
            "store_id" => $store_id,
            "next_num" => $next_num,
            "start_at" => $start_at,
            "return_all" => $return_all,
        ]);
    }

    /**
     * 根据门店id获取一组门店信息
     *
     * @param  array $store_ids 门店ID数组 [Original type array&lt;string&gt;] [ require true ]
     * @param  string $type 获取类型 [ require true ]
     * @param  bool $includeDelete 包含删除的店铺 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function getStoreInfoByIds (array $store_ids, string $type = 'base', bool $includeDelete = false) {
        return self::getClass()->getStoreInfoByIds([
            "store_ids" => $store_ids,
            "type" => $type,
            "includeDelete" => $includeDelete,
        ]);
    }

    /**
     * 根据门店id获取一组门店开业顺序信息
     *
     * @param  array $store_ids 门店ID数组 [Original type array&lt;uint&gt;] [ require true ]
     * @param  array $store_types 排序相关类型 [Original type array&lt;string&gt;] [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function getStoreOpenSort (array $store_ids, array $store_types) {
        return self::getClass()->getStoreOpenSort([
            "store_ids" => $store_ids,
            "store_types" => $store_types,
        ]);
    }

    /**
     * 根据门店id获取门店信息
     *
     * @param  int $store_id 门店ID [Original type uint] [ require true ]
     * @param  string $type 获取类型 [ require true ]
     * @param  bool $includeDelete 包含删除的店铺 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function getStoreInfoById (int $store_id, string $type = 'base', bool $includeDelete = false) {
        return self::getClass()->getStoreInfoById([
            "store_id" => $store_id,
            "type" => $type,
            "includeDelete" => $includeDelete,
        ]);
    }

    /**
     * 获取店铺列表(可进行搜索)
     *
     * @param  bool $includeDelete 包含删除的店铺 [ require true ]
     * @support  string $status 门店状态 
     * @param  int $page 当前的页码 [ require true ]
     * @support  string $name 店铺名(模糊查询) 
     * @support  int $groupId 集团ID [Original type uint] 
     * @support  int $provinceId 省份ID [Original type uint] 
     * @support  int $cityId 城市ID [Original type uint] 
     * @support  array $storeIds 门店ID [Original type array&lt;uint&gt;] 
     * @support  string $affiliateType 加盟类型 
     * @param  string $type 获取类型 [ require true ]
     * @support  array $storeTypes 门店类型 [Original type array&lt;string&gt;] 
     * @throws  \Exception
     * @return  array
     */
    public static function getStores (int $page = 1, bool $includeDelete = false, string $type = 'base') {
        return self::getClass()->getStores([
            "page" => $page,
            "includeDelete" => $includeDelete,
            "type" => $type,
        ]);
    }

    /**
     * 获取店铺列表(可进行搜索)
     *
     * @param  bool $includeDelete 包含删除的店铺 [ require true ]
     * @support  array $provinceIds 省份ID 
     * @support  array $cityIds 城市ID 
     * @support  array $storeIds 门店ID [Original type array&lt;uint&gt;] 
     * @support  array $affiliateTypes 加盟类型 
     * @param  string $type 获取类型 [ require true ]
     * @support  array $storeTypes 门店类型 [Original type array&lt;string&gt;] 
     * @throws  \Exception
     * @return  array
     */
    public static function getStoresWithoutPage (bool $includeDelete = false, string $type = 'base') {
        return self::getClass()->getStoresWithoutPage([
            "includeDelete" => $includeDelete,
            "type" => $type,
        ]);
    }

    /**
     * 根据门店名称获取门店信息
     *
     * @param  string $store_name 门店名称 [ require true ]
     * @param  string $type 获取类型 [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function getStoreInfoByName (string $store_name, string $type = 'base') {
        return self::getClass()->getStoreInfoByName([
            "store_name" => $store_name,
            "type" => $type,
        ]);
    }

    /**
     * 区间为[areaMin, areaMax)
     *
     * @param  int $areaMin 区间最小值, 将包含此值 [Original type uint] [ require true ]
     * @param  int $areaMax 区间最大值, 将不包含此值 [Original type uint] [ require true ]
     * @throws  \Exception
     * @return  int
     */
    public static function getMaxStoreIdInArea (int $areaMin, int $areaMax) {
        return self::getClass()->getMaxStoreIdInArea([
            "areaMin" => $areaMin,
            "areaMax" => $areaMax,
        ]);
    }

    /**
     * 创建门店
     *
     * @support  int $storeId 门店编号 [Original type uint] [ require true ]
     * @support  int $cityId 城市ID [Original type uint] [ require true ]
     * @support  int $provinceId 省ID [Original type uint] [ require true ]
     * @support  int $groupId 所属集团ID [Original type uint] 
     * @support  string $affiliateType 加盟类型 [ require true ]
     * @support  string $storeType 门店类型 [ require true ]
     * @support  string $name 门店名称 [ require true ]
     * @support  string $startAppointmentTime 开始营业时间 [ require true ]
     * @support  string $stopAppointmentTime 结束营业时间 [ require true ]
     * @support  int $serviceInterval 服务间隔(分钟) [Original type uint] [ require true ]
     * @support  int $serviceTimes 每间隔同时服务数 [Original type uint] [ require true ]
     * @support  string $startBusiness 开始营业时间 [ require true ]
     * @support  array $extend 拓展信息 [ require true ]
     * @support  string $dianpingId 大众点评门店ID 
     * @support  int $sort 门店排序 [Original type uint] [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function createStore () {
        return self::getClass()->createStore();
    }

    /**
     * 修改门店
     *
     * @param  int $storeId 门店编号 [Original type uint] [ require true ]
     * @support  int $cityId 城市ID [Original type uint] 
     * @support  string $affiliateType 加盟类型 
     * @support  string $name 门店名称 
     * @support  string $startAppointmentTime 开始营业时间 
     * @support  string $stopAppointmentTime 结束营业时间 
     * @support  array $extend 拓展信息 
     * @support  string $dianpingId 大众点评门店ID 
     * @support  int $sort 门店排序 [Original type uint] 
     * @support  string $usePosStatus pos机启用状态 
     * @support  int $groupId 所属集团ID [Original type uint] 
     * @support  int $provinceId 省ID [Original type uint] 
     * @throws  \Exception
     * @return  array
     */
    public static function editStore (int $storeId) {
        return self::getClass()->editStore([
            "storeId" => $storeId,
        ]);
    }

    /**
     * 修改门店开业时间
     *
     * @param  int $storeId 门店编号 [Original type uint] [ require true ]
     * @param  string $startBusiness 开始营业时间 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function editStoreStartBusiness (int $storeId, string $startBusiness) {
        return self::getClass()->editStoreStartBusiness([
            "storeId" => $storeId,
            "startBusiness" => $startBusiness,
        ]);
    }

    /**
     * 修改门店每时段服务数
     *
     * @param  int $storeId 门店编号 [Original type uint] [ require true ]
     * @param  int $changeTo 新的每时段服务数 [Original type uint] [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function editStoreServiceTimes (int $storeId, int $changeTo) {
        return self::getClass()->editStoreServiceTimes([
            "storeId" => $storeId,
            "changeTo" => $changeTo,
        ]);
    }

    /**
     * 修改门店每次服务间隔(分钟)
     *
     * @param  int $storeId 门店编号 [Original type uint] [ require true ]
     * @param  int $newServiceInterval 服务间隔 [Original type uint] [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function editStoreServiceInterval (int $storeId, int $newServiceInterval) {
        return self::getClass()->editStoreServiceInterval([
            "storeId" => $storeId,
            "newServiceInterval" => $newServiceInterval,
        ]);
    }

    /**
     * 通常用于下线操作
     *
     * @param  int $storeId 门店编号 [Original type uint] [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function deleteStore (int $storeId) {
        return self::getClass()->deleteStore([
            "storeId" => $storeId,
        ]);
    }

    /**
     * 通常用于上线操作
     *
     * @param  int $storeId 门店编号 [Original type uint] [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function restoreStore (int $storeId) {
        return self::getClass()->restoreStore([
            "storeId" => $storeId,
        ]);
    }

    /**
     * 根据点评ID换取门店ID
     *
     * @param  string $dianpingId 点评门店ID [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function getStoreIdByDianpingId (string $dianpingId) {
        return self::getClass()->getStoreIdByDianpingId([
            "dianpingId" => $dianpingId,
        ]);
    }

    /**
     * 根据海马体门店ID换取点评ID
     *
     * @param  int $storeId 海马体门店ID [Original type uint] [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function getDianpingIdByStoreId (int $storeId) {
        return self::getClass()->getDianpingIdByStoreId([
            "storeId" => $storeId,
        ]);
    }

    /**
     * getClass
     * @deprecated
     * @throws  \Exception
     * @return  mixed
     */
    public static function getClass () {
        return RpcClass::getClass('Store', 'Store');
    }

    /**
     * registerMock
     * @throws  \Exception
     * @return  RpcMockClass
     */
    public static function registerMock () {
        return RpcClass::registerMockClass('Store', 'Store');
    }

    /**
     * @param  $name
     * @param  $arguments
     * @return  mixed
     * @throws  \Exception
     */
    public function __call ($name, $arguments) {
        if (substr($name, -8, 8) === 'ByParams') {
            $shortName = str_replace('ByParams', '', $name);
            return self::getClass()->$shortName(...$arguments);
        }

        throw new InvalidArgumentException('Method not found! : '.$name);
    }
}