<?php
namespace Mainto\Bridge\Invokes\Product\Retail;

use Mainto\RpcServer\RpcClient\RpcClass;
use Mainto\RpcServer\RpcClient\RpcMockClass;
use InvalidArgumentException;

/**
 * Class Unique
 * @method createByParams(array $params)
 * @method updateByParams(array $params)
 * @method detailByParams(array $params)
 * @method getDetailByIdsByParams(array $params)
 * @method removeByParams(array $params)
 * @method listByParams(array $params)
 * @method allByParams(array $params)
 * @method upByParams(array $params)
 * @method offByParams(array $params)
 * @method getSkuByProductCodeByParams(array $params)
 * @method skuGroupsListByParams(array $params)
 * @method skuLabelsListByParams(array $params)
 * @method allSkuBasicByParams(array $params)
 * @method allSkuGroupByParams(array $params)
 * @method allSkuLabelByParams(array $params)
 * @method productStoresByParams(array $params)
 * @method judgeProductOfStoreByParams(array $params)
 */
class Product {
    private static $instance;

    /**
     * 获取一个远程调用类
     * @return  static
     */
    public static function getDirectClass () {
        if (self::$instance == null) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    /**
     * 创建产品
     *
     * @support  string $name 产品名称 [ require true ]
     * @support  int $category_id 分类ID [ require true ]
     * @support  string $main_img 产品主图 
     * @support  array $sku_basic 商品sku属性 [ require true ]
     * @support  array $sku 商品sku [ require true ]
     * @support  array $store_ids 关联店铺ids 
     * @throws  \Exception
     * @return  mixed
     */
    public static function create () {
        return self::getClass()->create();
    }

    /**
     * 编辑产品
     *
     * @support  int $id 产品id [ require true ]
     * @support  string $name 产品名称 [ require true ]
     * @support  int $category_id 分类ID 
     * @support  string $main_img 产品主图 
     * @support  array $sku_basic 商品sku属性 
     * @support  array $sku 商品sku 
     * @support  array $store_ids 关联店铺ids 
     * @throws  \Exception
     * @return  mixed
     */
    public static function update () {
        return self::getClass()->update();
    }

    /**
     * 产品详情
     *
     * @param  int $id 产品ID [Original type uint] [ require true ]
     * @param  string $status 产品状态 [ require true ]
     * @param  int $store_id 店铺id [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function detail (int $id, string $status = '', int $store_id = 0) {
        return self::getClass()->detail([
            "id" => $id,
            "status" => $status,
            "store_id" => $store_id,
        ]);
    }

    /**
     * 多个产品详情
     *
     * @param  array $ids 产品ids [ require true ]
     * @param  string $status 产品状态 [ require true ]
     * @param  int $store_id 店铺id [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function getDetailByIds (array $ids, string $status = '', int $store_id = 0) {
        return self::getClass()->getDetailByIds([
            "ids" => $ids,
            "status" => $status,
            "store_id" => $store_id,
        ]);
    }

    /**
     * 删除产品
     *
     * @param  int $id 产品ID [Original type uint] [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function remove (int $id) {
        return self::getClass()->remove([
            "id" => $id,
        ]);
    }

    /**
     * 获取产品分页列表
     *
     * @param  int $page 当前页 [ require true ]
     * @param  int $page_size 页内条数 [ require true ]
     * @param  string $name 名称(模糊查询) [ require true ]
     * @param  int $category_id 分类ID [ require true ]
     * @param  bool $show_deleted 是否显示已删除信息 [ require true ]
     * @param  bool $show_sku_deleted 是否显示已删除Sku信息 [ require true ]
     * @param  string $status 状态 [ require true ]
     * @param  array $store_ids 店铺ids [ require true ]
     * @param  array $area_ids 城市ids [ require true ]
     * @param  string $store_type 门店类型 [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function list (int $page = 1, int $page_size = 20, int $category_id = 0, string $name = '', string $status = '', string $store_type = '', array $store_ids = [], array $area_ids = [], bool $show_deleted = false, bool $show_sku_deleted = false) {
        return self::getClass()->list([
            "page" => $page,
            "page_size" => $page_size,
            "category_id" => $category_id,
            "name" => $name,
            "status" => $status,
            "store_type" => $store_type,
            "store_ids" => $store_ids,
            "area_ids" => $area_ids,
            "show_deleted" => $show_deleted,
            "show_sku_deleted" => $show_sku_deleted,
        ]);
    }

    /**
     * 获取产品列表
     *
     * @param  string $name 名称(模糊查询) [ require true ]
     * @param  int $category_id 分类ID [ require true ]
     * @param  bool $show_deleted 是否显示已删除信息 [ require true ]
     * @param  string $status 状态 [ require true ]
     * @param  array $store_ids 店铺ids [ require true ]
     * @param  array $area_ids 城市ids [ require true ]
     * @param  string $store_type 门店类型 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function all (int $category_id = 0, string $name = '', string $status = '', string $store_type = '', array $store_ids = [], array $area_ids = [], bool $show_deleted = false) {
        return self::getClass()->all([
            "category_id" => $category_id,
            "name" => $name,
            "status" => $status,
            "store_type" => $store_type,
            "store_ids" => $store_ids,
            "area_ids" => $area_ids,
            "show_deleted" => $show_deleted,
        ]);
    }

    /**
     * 上架产品
     *
     * @param  int $id 产品ID [Original type uint] [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function up (int $id) {
        return self::getClass()->up([
            "id" => $id,
        ]);
    }

    /**
     * 下架产品
     *
     * @param  int $id 产品ID [Original type uint] [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function off (int $id) {
        return self::getClass()->off([
            "id" => $id,
        ]);
    }

    /**
     * 根据编码获取sku信息
     *
     * @param  string $product_code 产品编码 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function getSkuByProductCode (string $product_code) {
        return self::getClass()->getSkuByProductCode([
            "product_code" => $product_code,
        ]);
    }

    /**
     * 获取分页列表
     *
     * @param  int $page 当前页 [ require true ]
     * @param  int $page_size 页内条数 [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function skuGroupsList (int $page = 1, int $page_size = 20) {
        return self::getClass()->skuGroupsList([
            "page" => $page,
            "page_size" => $page_size,
        ]);
    }

    /**
     * 获取分页列表
     *
     * @param  int $page 当前页 [ require true ]
     * @param  int $page_size 页内条数 [ require true ]
     * @support  bool $show_deleted 是否显示已删除信息 
     * @throws  \Exception
     * @return  mixed
     */
    public static function skuLabelsList (int $page = 1, int $page_size = 20) {
        return self::getClass()->skuLabelsList([
            "page" => $page,
            "page_size" => $page_size,
        ]);
    }

    /**
     * 获取所有sku_basic
     *
     * @throws  \Exception
     * @return  array
     */
    public static function allSkuBasic () {
        return self::getClass()->allSkuBasic();
    }

    /**
     * 获取所有sku分组
     *
     * @throws  \Exception
     * @return  array
     */
    public static function allSkuGroup () {
        return self::getClass()->allSkuGroup();
    }

    /**
     * 获取所有sku属性
     *
     * @throws  \Exception
     * @return  array
     */
    public static function allSkuLabel () {
        return self::getClass()->allSkuLabel();
    }

    /**
     * 产品关联的店铺信息
     *
     * @param  int $id 产品ID [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function productStores (int $id) {
        return self::getClass()->productStores([
            "id" => $id,
        ]);
    }

    /**
     * 校验零售商品是否在店铺中
     *
     * @param  int $id 产品ID [Original type uint] [ require true ]
     * @param  int $store_id 店铺id [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function judgeProductOfStore (int $id, int $store_id) {
        return self::getClass()->judgeProductOfStore([
            "id" => $id,
            "store_id" => $store_id,
        ]);
    }

    /**
     * getClass
     * @deprecated
     * @throws  \Exception
     * @return  mixed
     */
    public static function getClass () {
        return RpcClass::getClass('Product', 'Retail\Product');
    }

    /**
     * registerMock
     * @throws  \Exception
     * @return  RpcMockClass
     */
    public static function registerMock () {
        return RpcClass::registerMockClass('Product', 'Retail\Product');
    }

    /**
     * @param  $name
     * @param  $arguments
     * @return  mixed
     * @throws  \Exception
     */
    public function __call ($name, $arguments) {
        if (substr($name, -8, 8) === 'ByParams') {
            $shortName = str_replace('ByParams', '', $name);
            return self::getClass()->$shortName(...$arguments);
        }

        throw new InvalidArgumentException('Method not found! : '.$name);
    }
}