<?php
namespace Mainto\Bridge\Invokes\Product;

use Mainto\RpcServer\RpcClient\RpcClass;
use InvalidArgumentException;

/**
 * Class Unique
 * @method getTreeChildrenByParams(array $params)
 * @method getChildrenByParams(array $params)
 * @method getOnlyNextChildrenByParams(array $params)
 * @method addChildNodeByParams(array $params)
 * @method editNodeByParams(array $params)
 * @method getNodeByParams(array $params)
 * @method deleteNodeByParams(array $params)
 * @method restoreNodeByParams(array $params)
 * @method hideNodeByParams(array $params)
 * @method showNodeByParams(array $params)
 * @method batchAddNodeByParams(array $params)
 * @method getNodeNameByParams(array $params)
 * @method getNodeFullNameByParams(array $params)
 * @method moveNodeByParams(array $params)
 */
class ProductTree {
    private static $instance;

    /**
     * 获取一个远程调用类
     * @return  static
     */
    public static function getDirectClass () {
        if (self::$instance == null) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    /**
     * 树状模式获取下级节点
     *
     * @param  int $nodeId 节点ID [Original type uint] [ require true ]
     * @param  array $nodeType 节点类型 [ require true ]
     * @param  bool $includeHidden 包含隐藏节点 [ require true ]
     * @param  bool $includeDelete 包含下线节点 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function getTreeChildren (int $nodeId, array $nodeType = [0 => 'product'], bool $includeHidden = false, bool $includeDelete = false) {
        return self::getClass()->getTreeChildren([
            "nodeId" => $nodeId,
            "nodeType" => $nodeType,
            "includeHidden" => $includeHidden,
            "includeDelete" => $includeDelete,
        ]);
    }

    /**
     * 获取下级节点
     *
     * @param  int $nodeId 节点ID [Original type uint] [ require true ]
     * @param  array $nodeType 节点类型 [ require true ]
     * @param  bool $includeHidden 包含隐藏节点 [ require true ]
     * @param  bool $includeDelete 包含下线节点 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function getChildren (int $nodeId, array $nodeType = [0 => 'product'], bool $includeHidden = false, bool $includeDelete = false) {
        return self::getClass()->getChildren([
            "nodeId" => $nodeId,
            "nodeType" => $nodeType,
            "includeHidden" => $includeHidden,
            "includeDelete" => $includeDelete,
        ]);
    }

    /**
     * 获取下一级节点
     *
     * @param  int $nodeId 节点ID [Original type uint] [ require true ]
     * @param  array $nodeType 节点类型 [ require true ]
     * @param  bool $includeHidden 包含隐藏节点 [ require true ]
     * @param  bool $includeDelete 包含下线节点 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function getOnlyNextChildren (int $nodeId, array $nodeType = [0 => 'product'], bool $includeHidden = false, bool $includeDelete = false) {
        return self::getClass()->getOnlyNextChildren([
            "nodeId" => $nodeId,
            "nodeType" => $nodeType,
            "includeHidden" => $includeHidden,
            "includeDelete" => $includeDelete,
        ]);
    }

    /**
     * 添加下级节点
     *
     * @param  int $parentNodeId 父节点 [Original type uint] [ require true ]
     * @param  string $nodeType 节点类型 [ require true ]
     * @param  bool $isHidden 是否为隐藏节点 [ require true ]
     * @param  string $name 节点名称 [ require true ]
     * @param  string $engName 节点英文名称 [ require true ]
     * @param  int $includeNum 包含子产品数量, 仅 AdditionNode 有效, 0或-1为不强制包含子产品 [Original type uint] [ require true ]
     * @param  string $fullName 完整名称 [ require true ]
     * @param  int $minPeople 最小人数 [Original type uint] [ require true ]
     * @param  int $maxPeople 最大人数 [Original type uint] [ require true ]
     * @param  string $realName 节点真实名称 [ require true ]
     * @param  int $sort 排序顺序 [Original type uint] [ require true ]
     * @param  array $extend 拓展信息 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function addChildNode (int $parentNodeId, string $nodeType, string $name, string $engName, int $includeNum = 0, string $fullName = 'auto', int $minPeople = 1, int $maxPeople = 1, array $extend = [], int $sort = 0, string $realName = '', bool $isHidden = false) {
        return self::getClass()->addChildNode([
            "parentNodeId" => $parentNodeId,
            "nodeType" => $nodeType,
            "name" => $name,
            "engName" => $engName,
            "includeNum" => $includeNum,
            "fullName" => $fullName,
            "minPeople" => $minPeople,
            "maxPeople" => $maxPeople,
            "extend" => $extend,
            "sort" => $sort,
            "realName" => $realName,
            "isHidden" => $isHidden,
        ]);
    }

    /**
     * 修改下级节点
     *
     * @param  int $nodeId 父节点 [Original type uint] [ require true ]
     * @param  string $name 节点名称 [ require true ]
     * @param  string $engName 节点英文名称 [ require true ]
     * @param  int $includeNum 包含子产品数量, 仅 AdditionNode 有效, -1为不更改, 0为不包含子产品 [Original type uint] [ require true ]
     * @param  string $fullName 完整名称 [ require true ]
     * @param  int $minPeople 最小人数 [Original type uint] [ require true ]
     * @param  int $maxPeople 最大人数 [Original type uint] [ require true ]
     * @param  string $realName 节点真实名称 [ require true ]
     * @param  int $sort 排序顺序 [Original type uint] [ require true ]
     * @param  array $extend 拓展信息 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function editNode (int $nodeId, int $sort = 0, array $extend = [], string $name = '', string $engName = '', int $includeNum = -1, string $fullName = 'auto', int $minPeople = 1, int $maxPeople = 1, string $realName = '') {
        return self::getClass()->editNode([
            "nodeId" => $nodeId,
            "sort" => $sort,
            "extend" => $extend,
            "name" => $name,
            "engName" => $engName,
            "includeNum" => $includeNum,
            "fullName" => $fullName,
            "minPeople" => $minPeople,
            "maxPeople" => $maxPeople,
            "realName" => $realName,
        ]);
    }

    /**
     * 获取某个节点
     *
     * @param  int $nodeId 节点ID [Original type uint] [ require true ]
     * @param  bool $includeHidden 包含隐藏节点 [ require true ]
     * @param  bool $includeDelete 包含下线节点 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function getNode (int $nodeId, bool $includeHidden = false, bool $includeDelete = false) {
        return self::getClass()->getNode([
            "nodeId" => $nodeId,
            "includeHidden" => $includeHidden,
            "includeDelete" => $includeDelete,
        ]);
    }

    /**
     * 通常用于下线
     * 当有父级下线时, 其子项将会被影响成为下线状态(此下线状态不会在数据库中更改, 只是逻辑上的覆盖)
     *
     * @param  int $nodeId 节点ID [Original type uint] [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function deleteNode (int $nodeId) {
        return self::getClass()->deleteNode([
            "nodeId" => $nodeId,
        ]);
    }

    /**
     * 通常用于上线
     *
     * @param  int $nodeId 节点ID [Original type uint] [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function restoreNode (int $nodeId) {
        return self::getClass()->restoreNode([
            "nodeId" => $nodeId,
        ]);
    }

    /**
     * 隐藏后该产品将不能直接获取到, 但是依旧应该可以售卖
     * 当有父级隐藏时, 其子项将会被影响成为隐藏状态(此下线状态不会在数据库中更改, 只是逻辑上的覆盖)
     *
     * @param  int $nodeId 节点ID [Original type uint] [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function hideNode (int $nodeId) {
        return self::getClass()->hideNode([
            "nodeId" => $nodeId,
        ]);
    }

    /**
     * 是隐藏节点的相反操作
     *
     * @param  int $nodeId 节点ID [Original type uint] [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function showNode (int $nodeId) {
        return self::getClass()->showNode([
            "nodeId" => $nodeId,
        ]);
    }

    /**
     * nodes的格式是一个多层的数组, 可以参照 getTreeChildren 返回的样式
     * 允许的字段包含 node_type, include_num, is_hidden, name, real_name, sort, extend, children
     * 其中 name 和 node_type 必填, include_num 只能在 AdditionNode 节点中使用
     *
     * @param  int $parentNodeId 父节点 [Original type uint] [ require true ]
     * @param  array $node 节点信息 [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function batchAddNode (int $parentNodeId, array $node) {
        return self::getClass()->batchAddNode([
            "parentNodeId" => $parentNodeId,
            "node" => $node,
        ]);
    }

    /**
     * 获取某个节点名称
     *
     * @param  int $nodeId 节点ID [Original type uint] [ require true ]
     * @param  bool $includeHidden 包含隐藏节点 [ require true ]
     * @param  bool $includeDelete 包含下线节点 [ require true ]
     * @throws  \Exception
     * @return  string
     */
    public static function getNodeName (int $nodeId, bool $includeHidden = false, bool $includeDelete = false) {
        return self::getClass()->getNodeName([
            "nodeId" => $nodeId,
            "includeHidden" => $includeHidden,
            "includeDelete" => $includeDelete,
        ]);
    }

    /**
     * 获取某个节点完整名称
     *
     * @param  int $nodeId 节点ID [Original type uint] [ require true ]
     * @param  bool $includeHidden 包含隐藏节点 [ require true ]
     * @param  bool $includeDelete 包含下线节点 [ require true ]
     * @throws  \Exception
     * @return  string
     */
    public static function getNodeFullName (int $nodeId, bool $includeHidden = false, bool $includeDelete = false) {
        return self::getClass()->getNodeFullName([
            "nodeId" => $nodeId,
            "includeHidden" => $includeHidden,
            "includeDelete" => $includeDelete,
        ]);
    }

    /**
     * 移动节点到新的父级下
     *
     * @param  int $nodeId 节点ID [Original type uint] [ require true ]
     * @param  int $newParentId 节点ID [Original type uint] [ require true ]
     * @param  bool $genNewFullName 生成新的名称 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function moveNode (int $nodeId, int $newParentId, bool $genNewFullName = false) {
        return self::getClass()->moveNode([
            "nodeId" => $nodeId,
            "newParentId" => $newParentId,
            "genNewFullName" => $genNewFullName,
        ]);
    }

    /**
     * getClass
     * @deprecated
     * @throws  \Exception
     * @return  mixed
     */
    public static function getClass () {
        return RpcClass::getClass('Product', 'ProductTree');
    }

    /**
     * @param  $name
     * @param  $arguments
     * @return  mixed
     * @throws  \Exception
     */
    public function __call ($name, $arguments) {
        if (substr($name, -8, 8) === 'ByParams') {
            $shortName = str_replace('ByParams', '', $name);
            return self::getClass()->$shortName(...$arguments);
        }

        throw new InvalidArgumentException('Method not found! : '.$name);
    }
}