<?php
namespace Mainto\Bridge\Invokes\Product;

use Mainto\RpcServer\RpcClient\RpcClass;
use InvalidArgumentException;

/**
 * Class Unique
 * @method getStorePriceByParams(array $params)
 * @method getProductPriceByParams(array $params)
 * @method addProductPriceByParams(array $params)
 * @method bindStoreAndProductPriceByParams(array $params)
 * @method deleteStoreProductPriceByParams(array $params)
 * @method getProductPriceStoreByParams(array $params)
 * @method deleteAllStoreProductPriceByParams(array $params)
 * @method deleteProductAllStoreAllPriceByParams(array $params)
 */
class ProductPrice {
    private static $instance;

    /**
     * 获取一个远程调用类
     * @return  static
     */
    public static function getDirectClass () {
        if (self::$instance == null) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    /**
     * 获取某家门店的产品价格
     *
     * @param  int $storeId 门店ID [Original type uint] [ require true ]
     * @param  array $productIds 限制查询的产品ID [Original type array&lt;uint&gt;] [ require true ]
     * @param  bool $includeHidden 包含隐藏节点 [ require true ]
     * @param  bool $includeDelete 包含下线节点 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function getStorePrice (int $storeId, array $productIds = [], bool $includeHidden = false, bool $includeDelete = false) {
        return self::getClass()->getStorePrice([
            "storeId" => $storeId,
            "productIds" => $productIds,
            "includeHidden" => $includeHidden,
            "includeDelete" => $includeDelete,
        ]);
    }

    /**
     * 获取产品价格
     *
     * @param  int $nodeId 节点ID [Original type uint] [ require true ]
     * @param  bool $includeHidden 包含隐藏节点 [ require true ]
     * @param  bool $includeDelete 包含下线节点 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function getProductPrice (int $nodeId, bool $includeHidden = false, bool $includeDelete = false) {
        return self::getClass()->getProductPrice([
            "nodeId" => $nodeId,
            "includeHidden" => $includeHidden,
            "includeDelete" => $includeDelete,
        ]);
    }

    /**
     * 为某个节点创建一个新的价格
     *
     * @param  int $nodeId 节点ID [Original type uint] [ require true ]
     * @param  float $price 售卖价格 [Original type unsignedDouble] [ require true ]
     * @param  float $originalPrice 原价 [Original type unsignedDouble] [ require true ]
     * @param  float $addPeoplePrice 加人价格 [Original type unsignedDouble] [ require true ]
     * @param  string $priceMode 价格计算方式 [ require true ]
     * @param  array $extend 拓展信息 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function addProductPrice (int $nodeId, float $price, float $originalPrice, float $addPeoplePrice, string $priceMode = 'single', array $extend = []) {
        return self::getClass()->addProductPrice([
            "nodeId" => $nodeId,
            "price" => $price,
            "originalPrice" => $originalPrice,
            "addPeoplePrice" => $addPeoplePrice,
            "priceMode" => $priceMode,
            "extend" => $extend,
        ]);
    }

    /**
     * 一旦某家门店有某个产品的价格, 该产品将在此家门店可售卖
     *
     * @param  int $priceId 价格ID [Original type uint] [ require true ]
     * @param  array $storeIds 门店ID [Original type array&lt;uint&gt;] [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function bindStoreAndProductPrice (int $priceId, array $storeIds) {
        return self::getClass()->bindStoreAndProductPrice([
            "priceId" => $priceId,
            "storeIds" => $storeIds,
        ]);
    }

    /**
     * 一旦某家门店没有某个产品, 该产品将在此家门店不可售卖
     *
     * @param  int $priceId 价格ID [Original type uint] [ require true ]
     * @param  array $storeIds 门店ID [Original type array&lt;uint&gt;] [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function deleteStoreProductPrice (int $priceId, array $storeIds) {
        return self::getClass()->deleteStoreProductPrice([
            "priceId" => $priceId,
            "storeIds" => $storeIds,
        ]);
    }

    /**
     * 获取某个价格绑定的所有门店
     *
     * @param  int $priceId 价格ID [Original type uint] [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function getProductPriceStore (int $priceId) {
        return self::getClass()->getProductPriceStore([
            "priceId" => $priceId,
        ]);
    }

    /**
     * 删除某个价格绑定的所有门店
     *
     * @param  int $priceId 价格ID [Original type uint] [ require true ]
     * @param  bool $deleteSelf 删除价格本身 [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function deleteAllStoreProductPrice (int $priceId, bool $deleteSelf = false) {
        return self::getClass()->deleteAllStoreProductPrice([
            "priceId" => $priceId,
            "deleteSelf" => $deleteSelf,
        ]);
    }

    /**
     * 删除某个产品的价格绑定的所有门店
     *
     * @param  int $nodeId 价格ID [Original type uint] [ require true ]
     * @param  bool $deleteSelf 删除价格本身 [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function deleteProductAllStoreAllPrice (int $nodeId, bool $deleteSelf = false) {
        return self::getClass()->deleteProductAllStoreAllPrice([
            "nodeId" => $nodeId,
            "deleteSelf" => $deleteSelf,
        ]);
    }

    /**
     * getClass
     * @deprecated
     * @throws  \Exception
     * @return  mixed
     */
    public static function getClass () {
        return RpcClass::getClass('Product', 'ProductPrice');
    }

    /**
     * @param  $name
     * @param  $arguments
     * @return  mixed
     * @throws  \Exception
     */
    public function __call ($name, $arguments) {
        if (substr($name, -8, 8) === 'ByParams') {
            $shortName = str_replace('ByParams', '', $name);
            return self::getClass()->$shortName(...$arguments);
        }

        throw new InvalidArgumentException('Method not found! : '.$name);
    }
}