<?php
namespace Mainto\Bridge\Invokes\Preferential\GiftCard;

use Mainto\RpcServer\RpcClient\RpcClass;
use Mainto\RpcServer\RpcClient\RpcMockClass;
use InvalidArgumentException;

/**
 * Class Unique
 * @method batchCreateCardByParams(array $params)
 * @method createRefundCardByParams(array $params)
 * @method getCardInfoByParams(array $params)
 * @method getCardListByParams(array $params)
 * @method getCardListByPageByParams(array $params)
 * @method getUserBalanceByParams(array $params)
 * @method getUserValidDateByParams(array $params)
 * @method chargeByParams(array $params)
 * @method bindCardByParams(array $params)
 * @method abolishCardByParams(array $params)
 * @method transCardToWXCardByParams(array $params)
 * @method bindAndRemoveWXCardByParams(array $params)
 * @method getUserGiftCardPaymentLogsByParams(array $params)
 * @method delayByParams(array $params)
 * @method withdrawByParams(array $params)
 */
class GiftCard {
    private static $instance;

    /**
     * 获取一个远程调用类
     * @return  static
     */
    public static function getDirectClass () {
        if (self::$instance == null) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    /**
     * 批量生成礼品卡，可以生成1-N张
     * 生成的礼品卡都是未激活的
     *
     * @support  string $title 记录标题 [ require true ]
     * @support  array $extend 拓展信息 
     * @support  int $create_by 创建人 [Original type uint] [ require true ]
     * @support  int $cover_id 礼品卡封面 [ require true ]
     * @support  int $card_money 礼品卡面值 [ require true ]
     * @support  int $count 生成数量 [ require true ]
     * @throws  \Exception
     * @return  int
     */
    public static function batchCreateCard () {
        return self::getClass()->batchCreateCard();
    }

    /**
     * 退款生成礼品卡（一张）
     * 这里主要用于满足将现金退款至新礼品卡的业务，是一种特殊场景
     * 生成的礼品卡不与订单关联，自动激活，这扣比例1:1
     * 生成记录也单独记录，不与礼品卡创建记录关联
     *
     * @param  float $money 记录标题 [Original type unsignedDouble] [ require true ]
     * @param  int $user_id 用户id [Original type uint] [ require true ]
     * @param  int $out_refund_id 退款id [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function createRefundCard (int $user_id, float $money, int $out_refund_id) {
        return self::getClass()->createRefundCard([
            "user_id" => $user_id,
            "money" => $money,
            "out_refund_id" => $out_refund_id,
        ]);
    }

    /**
     * 获取详情
     *
     * @support  int $id id [Original type uint] 
     * @support  string $code 礼品卡code 
     * @param  bool $with_cover_info 是否包含卡面信息 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function getCardInfo (bool $with_cover_info = false) {
        return self::getClass()->getCardInfo([
            "with_cover_info" => $with_cover_info,
        ]);
    }

    /**
     * 获取列表
     *
     * @support  array $ids 礼品卡编号 
     * @support  bool $bound 是否已绑定 
     * @support  int $user_id 用户id [Original type uint] 
     * @support  int $old_user_id 原持有用户（赠送前）id [Original type uint] 
     * @support  int $card_id 卡片id [Original type uint] 
     * @support  string $code 礼品码 
     * @support  array $codes 礼品码组 
     * @support  float $money 面额 [Original type unsignedDouble] 
     * @support  array $status 状态 
     * @support  array $give_status 赠送状态 
     * @support  int $create_log_id 生成记录id [Original type uint] 
     * @support  bool $with_cover_info 是否包含卡面信息 
     * @support  string $sort_field 排序字段 
     * @support  string $sort_direction 排序方向 
     * @support  string $created_at_start 生成时间起 
     * @support  string $created_at_end 生成时间止 
     * @throws  \Exception
     * @return  array
     */
    public static function getCardList () {
        return self::getClass()->getCardList();
    }

    /**
     * 获取列表-分页
     *
     * @support  array $ids 礼品卡ids 
     * @support  bool $bound 是否已绑定 
     * @support  array $status 状态 
     * @support  array $give_status 赠送状态 
     * @support  int $user_id 用户id [Original type uint] 
     * @support  int $old_user_id 原持有用户（赠送前）id [Original type uint] 
     * @support  string $code 礼品码 
     * @support  int $create_log_id 生成记录id [Original type uint] 
     * @support  bool $with_cover_info 是否包含卡面信息 
     * @support  string $created_at_start 生成时间起 
     * @support  string $created_at_end 生成时间止 
     * @support  array $order_nums 购买订单号 
     * @param  int $page 页码 [ require true ]
     * @param  int $page_size 每页条数 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function getCardListByPage (int $page = 1, int $page_size = 10) {
        return self::getClass()->getCardListByPage([
            "page" => $page,
            "page_size" => $page_size,
        ]);
    }

    /**
     * 获取用户余额
     *
     * @param  int $user_id 用户id [Original type uint] [ require true ]
     * @throws  \Exception
     * @return  int
     */
    public static function getUserBalance (int $user_id) {
        return self::getClass()->getUserBalance([
            "user_id" => $user_id,
        ]);
    }

    /**
     * 获取用户礼品卡有效期
     *
     * @param  int $user_id 用户id [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function getUserValidDate (int $user_id) {
        return self::getClass()->getUserValidDate([
            "user_id" => $user_id,
        ]);
    }

    /**
     * 充值
     *
     * @param  int $user_id 用户id [Original type uint] [ require true ]
     * @param  string $code 礼品码 [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function charge (int $user_id, string $code) {
        return self::getClass()->charge([
            "user_id" => $user_id,
            "code" => $code,
        ]);
    }

    /**
     * TODO 新礼品卡业务里，绑定还需要吗？
     *
     * @param  int $user_id 用户id [ require true ]
     * @param  string $code 礼品卡code [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function bindCard (int $user_id, string $code) {
        return self::getClass()->bindCard([
            "user_id" => $user_id,
            "code" => $code,
        ]);
    }

    /**
     * 作废礼品码
     *
     * @support  array $codes 礼品码 
     * @support  array $cardIds 礼品卡ids [Original type array&lt;int&gt;] 
     * @support  int $operator 操作人 [ require true ]
     * @support  array $extend 扩展信息 
     * @throws  \Exception
     * @return  mixed
     */
    public static function abolishCard () {
        return self::getClass()->abolishCard();
    }

    /**
     * 转出礼品卡到微信卡包
     *
     * @param  string $scene 通道 [ require true ]
     * @param  int $user_id 用户id [ require true ]
     * @param  string $open_id openid [ require true ]
     * @param  string $cardId 卡id [ require true ]
     * @param  string $code 礼品卡code [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function transCardToWXCard (string $scene, int $user_id, string $open_id, string $cardId, string $code) {
        return self::getClass()->transCardToWXCard([
            "scene" => $scene,
            "user_id" => $user_id,
            "open_id" => $open_id,
            "cardId" => $cardId,
            "code" => $code,
        ]);
    }

    /**
     * 绑定微信卡包礼品卡
     *
     * @param  string $scene 通道 [ require true ]
     * @param  int $user_id 用户id [ require true ]
     * @param  string $cardId 卡id [ require true ]
     * @param  string $code 礼品卡code [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function bindAndRemoveWXCard (string $scene, int $user_id, string $cardId, string $code) {
        return self::getClass()->bindAndRemoveWXCard([
            "scene" => $scene,
            "user_id" => $user_id,
            "cardId" => $cardId,
            "code" => $code,
        ]);
    }

    /**
     * 查询用户礼品卡消费记录
     *
     * @param  int $user_id 用户ID [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function getUserGiftCardPaymentLogs (int $user_id) {
        return self::getClass()->getUserGiftCardPaymentLogs([
            "user_id" => $user_id,
        ]);
    }

    /**
     * 礼品码延期
     *
     * @support  array $codes 礼品码 [ require true ]
     * @support  string $stop_usage 到期日期 [ require true ]
     * @support  int $operator 操作人 [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function delay () {
        return self::getClass()->delay();
    }

    /**
     * 注意：目前只有礼品卡转退（钱退款至礼品卡）一种场景支持提现，所以出于安全考虑，在这里做了限制，邀请必须有退款至新礼品卡的记录！！！
     * 后期如果完善礼品卡提现业务，可以对此进行重构
     *
     * @param  int $card_id 礼品卡ID [ require true ]
     * @param  float $money 提现金额 [Original type double] [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function withdraw (int $card_id, float $money) {
        return self::getClass()->withdraw([
            "card_id" => $card_id,
            "money" => $money,
        ]);
    }

    /**
     * getClass
     * @deprecated
     * @throws  \Exception
     * @return  mixed
     */
    public static function getClass () {
        return RpcClass::getClass('Preferential', 'GiftCard\GiftCard');
    }

    /**
     * registerMock
     * @throws  \Exception
     * @return  RpcMockClass
     */
    public static function registerMock () {
        return RpcClass::registerMockClass('Preferential', 'GiftCard\GiftCard');
    }

    /**
     * @param  $name
     * @param  $arguments
     * @return  mixed
     * @throws  \Exception
     */
    public function __call ($name, $arguments) {
        if (substr($name, -8, 8) === 'ByParams') {
            $shortName = str_replace('ByParams', '', $name);
            return self::getClass()->$shortName(...$arguments);
        }

        throw new InvalidArgumentException('Method not found! : '.$name);
    }
}