<?php
namespace Mainto\Bridge\Invokes\Preferential;

use Mainto\RpcServer\RpcClient\RpcClass;
use Mainto\RpcServer\RpcClient\RpcMockClass;
use InvalidArgumentException;

/**
 * Class Unique
 * @method genCardsByParams(array $params)
 * @method genRefundGiftCardByParams(array $params)
 * @method refundByParams(array $params)
 * @method updateCardsByParams(array $params)
 * @method activateCardByParams(array $params)
 * @method getCardInfoByParams(array $params)
 * @method getCardListByParams(array $params)
 * @method getCardListByPageByParams(array $params)
 * @method getPayLogsByParams(array $params)
 * @method getUserBalanceByParams(array $params)
 * @method getUserValidDateByParams(array $params)
 * @method useGiftCardByParams(array $params)
 * @method canRefundMoneyByParams(array $params)
 * @method chargeByParams(array $params)
 * @method transCardToWXCardByParams(array $params)
 * @method bindAndRemoveWXCardByParams(array $params)
 * @method bindCardByParams(array $params)
 * @method destroyCardByParams(array $params)
 * @method updatePayLogByParams(array $params)
 */
class UserGiftCard {
    private static $instance;

    /**
     * 获取一个远程调用类
     * @return  static
     */
    public static function getDirectClass () {
        if (self::$instance == null) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    /**
     * 管理端生成礼品卡
     *
     * @param  string $title 记录标题 [ require true ]
     * @param  int $create_by 创建人 [Original type uint] [ require true ]
     * @support  array $extend 拓展信息 
     * @param  int $channel 生成渠道[1:系统生成] [ require true ]
     * @support  string $start_usage 开始使用时间 
     * @support  string $stop_usage 结束使用时间 
     * @param  array $card_item 卡片信息 [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function genCards (int $create_by, string $title, array $card_item, int $channel) {
        return self::getClass()->genCards([
            "create_by" => $create_by,
            "title" => $title,
            "card_item" => $card_item,
            "channel" => $channel,
        ]);
    }

    /**
     * 退款生成礼品卡
     *
     * @param  float $money 记录标题 [Original type unsignedDouble] [ require true ]
     * @support  array $extend 拓展信息 
     * @param  int $user_id 用户id [Original type uint] [ require true ]
     * @param  string $user_name 用户名 [ require true ]
     * @support  string $phone 手机号 [ require true ]
     * @support  string $sex 性别 [ require true ]
     * @support  int $store_id 门店 [ require true ]
     * @param  int $refund_id 退款id [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function genRefundGiftCard (float $money, int $user_id, int $refund_id, string $user_name = '') {
        return self::getClass()->genRefundGiftCard([
            "money" => $money,
            "user_id" => $user_id,
            "refund_id" => $refund_id,
            "user_name" => $user_name,
        ]);
    }

    /**
     * 订单退款【礼品卡支付订单】
     *
     * @param  float $drawback_money 退款金额 [Original type double] [ require true ]
     * @param  string $order_num 订单号 [ require true ]
     * @param  string $pay_for 支付时用途(不填则订单+二销) [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function refund (float $drawback_money, string $order_num, string $pay_for = '') {
        return self::getClass()->refund([
            "drawback_money" => $drawback_money,
            "order_num" => $order_num,
            "pay_for" => $pay_for,
        ]);
    }

    /**
     * 更新
     *
     * @support  int $cover_id 卡面id [Original type uint] 
     * @support  string $status 状态 
     * @param  array $card_ids 卡ids [Original type array&lt;int&gt;] [ require true ]
     * @param  int $operator 操作者 [Original type uint] [ require true ]
     * @support  string $stop_usage 过期时间 
     * @throws  \Exception
     * @return  mixed
     */
    public static function updateCards (array $card_ids, int $operator) {
        return self::getClass()->updateCards([
            "card_ids" => $card_ids,
            "operator" => $operator,
        ]);
    }

    /**
     * 激活礼品卡【接口暂无可用】
     * @deprecated
     *
     * @param  array $card_ids 卡ids [Original type array&lt;int&gt;] [ require true ]
     * @param  int $operator 操作者 [Original type uint] [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function activateCard (array $card_ids, int $operator) {
        return self::getClass()->activateCard([
            "card_ids" => $card_ids,
            "operator" => $operator,
        ]);
    }

    /**
     * 获取详情
     *
     * @support  int $id id [Original type uint] 
     * @support  string $code 礼品卡code 
     * @param  bool $with_cover_info 是否包含卡面信息 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function getCardInfo (bool $with_cover_info = false) {
        return self::getClass()->getCardInfo([
            "with_cover_info" => $with_cover_info,
        ]);
    }

    /**
     * 获取列表
     *
     * @support  array $ids 礼品卡编号 
     * @support  bool $bound 是否已绑定 
     * @support  int $user_id 用户id [Original type uint] 
     * @support  int $card_id 卡片id [Original type uint] 
     * @support  string $code 礼品码 
     * @support  float $money 面额 [Original type unsignedDouble] 
     * @support  array $status 状态 
     * @support  int $create_log_id 生成记录id [Original type uint] 
     * @support  bool $with_cover_info 是否包含卡面信息 
     * @support  string $created_at_start 生成时间起 
     * @support  string $created_at_end 生成时间止 
     * @throws  \Exception
     * @return  array
     */
    public static function getCardList () {
        return self::getClass()->getCardList();
    }

    /**
     * 获取列表-分页
     *
     * @support  array $ids 礼品卡ids 
     * @support  bool $bound 是否已绑定 
     * @support  array $status 状态 
     * @support  int $user_id 用户id [Original type uint] 
     * @support  string $code 礼品码 
     * @support  int $create_log_id 生成记录id [Original type uint] 
     * @support  bool $with_cover_info 是否包含卡面信息 
     * @support  string $created_at_start 生成时间起 
     * @support  string $created_at_end 生成时间止 
     * @support  array $order_nums 购买订单号 
     * @param  int $page 页码 [ require true ]
     * @param  int $page_size 每页条数 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function getCardListByPage (int $page = 1, int $page_size = 10) {
        return self::getClass()->getCardListByPage([
            "page" => $page,
            "page_size" => $page_size,
        ]);
    }

    /**
     * 获取用户支付记录
     *
     * @param  int $user_id 用户id [Original type uint] [ require true ]
     * @param  string $order_num 订单号 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function getPayLogs (int $user_id = 0, string $order_num = '') {
        return self::getClass()->getPayLogs([
            "user_id" => $user_id,
            "order_num" => $order_num,
        ]);
    }

    /**
     * 获取用户余额
     *
     * @param  int $user_id 用户id [Original type uint] [ require true ]
     * @throws  \Exception
     * @return  int
     */
    public static function getUserBalance (int $user_id) {
        return self::getClass()->getUserBalance([
            "user_id" => $user_id,
        ]);
    }

    /**
     * 获取用户礼品卡有效期
     *
     * @param  int $user_id 用户id [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function getUserValidDate (int $user_id) {
        return self::getClass()->getUserValidDate([
            "user_id" => $user_id,
        ]);
    }

    /**
     * 礼品卡下单
     *
     * @param  int $user_id 用户id [Original type uint] [ require true ]
     * @param  float $money 支付金额 [Original type double] [ require true ]
     * @param  string $order_num 订单号 [ require true ]
     * @param  string $pay_for 用途 [ require true ]
     * @param  array $extend 拓展信息 [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function useGiftCard (int $user_id, float $money, string $order_num, string $pay_for, array $extend = []) {
        return self::getClass()->useGiftCard([
            "user_id" => $user_id,
            "money" => $money,
            "order_num" => $order_num,
            "pay_for" => $pay_for,
            "extend" => $extend,
        ]);
    }

    /**
     * 获取可退款礼品卡金额
     *
     * @param  string $pay_for 用途 [ require true ]
     * @param  string $order_num 订单号 [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function canRefundMoney (string $order_num, string $pay_for = '') {
        return self::getClass()->canRefundMoney([
            "order_num" => $order_num,
            "pay_for" => $pay_for,
        ]);
    }

    /**
     * 充值
     *
     * @param  int $user_id 用户id [Original type uint] [ require true ]
     * @param  string $code 礼品码 [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function charge (int $user_id, string $code) {
        return self::getClass()->charge([
            "user_id" => $user_id,
            "code" => $code,
        ]);
    }

    /**
     * 转出礼品卡到微信卡包
     *
     * @param  string $scene 通道 [ require true ]
     * @param  int $user_id 用户id [ require true ]
     * @param  string $open_id openid [ require true ]
     * @param  string $cardId 卡id [ require true ]
     * @param  string $code 礼品卡code [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function transCardToWXCard (string $scene, int $user_id, string $open_id, string $cardId, string $code) {
        return self::getClass()->transCardToWXCard([
            "scene" => $scene,
            "user_id" => $user_id,
            "open_id" => $open_id,
            "cardId" => $cardId,
            "code" => $code,
        ]);
    }

    /**
     * 绑定微信卡包礼品卡
     *
     * @param  string $scene 通道 [ require true ]
     * @param  int $user_id 用户id [ require true ]
     * @param  string $cardId 卡id [ require true ]
     * @param  string $code 礼品卡code [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function bindAndRemoveWXCard (string $scene, int $user_id, string $cardId, string $code) {
        return self::getClass()->bindAndRemoveWXCard([
            "scene" => $scene,
            "user_id" => $user_id,
            "cardId" => $cardId,
            "code" => $code,
        ]);
    }

    /**
     * 绑定礼品卡
     *
     * @param  int $user_id 用户id [ require true ]
     * @param  string $code 礼品卡code [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function bindCard (int $user_id, string $code) {
        return self::getClass()->bindCard([
            "user_id" => $user_id,
            "code" => $code,
        ]);
    }

    /**
     * 销毁礼品卡【退回退款至礼品卡的金额】
     *
     * @param  int $card_id 礼品卡id [ require true ]
     * @param  float $money 退回金额 [Original type unsignedDouble] [ require true ]
     * @param  int $user_id 用户id [ require true ]
     * @support  int $order_id 用户id 
     * @throws  \Exception
     * @return  mixed
     */
    public static function destroyCard (int $card_id, float $money, int $user_id) {
        return self::getClass()->destroyCard([
            "card_id" => $card_id,
            "money" => $money,
            "user_id" => $user_id,
        ]);
    }

    /**
     * 更新支付记录【转门店】
     *
     * @param  string $order_num 老订单号 [ require true ]
     * @param  string $new_order_num 新订单号 [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function updatePayLog (string $order_num, string $new_order_num) {
        return self::getClass()->updatePayLog([
            "order_num" => $order_num,
            "new_order_num" => $new_order_num,
        ]);
    }

    /**
     * getClass
     * @deprecated
     * @throws  \Exception
     * @return  mixed
     */
    public static function getClass () {
        return RpcClass::getClass('Preferential', 'UserGiftCard');
    }

    /**
     * registerMock
     * @throws  \Exception
     * @return  RpcMockClass
     */
    public static function registerMock () {
        return RpcClass::registerMockClass('Preferential', 'UserGiftCard');
    }

    /**
     * @param  $name
     * @param  $arguments
     * @return  mixed
     * @throws  \Exception
     */
    public function __call ($name, $arguments) {
        if (substr($name, -8, 8) === 'ByParams') {
            $shortName = str_replace('ByParams', '', $name);
            return self::getClass()->$shortName(...$arguments);
        }

        throw new InvalidArgumentException('Method not found! : '.$name);
    }
}