<?php
namespace Mainto\Bridge\Invokes\Order\Retail;

use Mainto\RpcServer\RpcClient\RpcClass;
use Mainto\RpcServer\RpcClient\RpcMockClass;
use InvalidArgumentException;

/**
 * Class Unique
 * @method createByParams(array $params)
 * @method updateByParams(array $params)
 * @method detailByParams(array $params)
 * @method detailByOrderNoByParams(array $params)
 * @method submitByParams(array $params)
 * @method removeByParams(array $params)
 * @method deletePaidByParams(array $params)
 * @method changePayTypeByParams(array $params)
 * @method listByParams(array $params)
 * @method setPaidByParams(array $params)
 * @method closeByParams(array $params)
 * @method changeUserDataByParams(array $params)
 * @method getIdByNoByParams(array $params)
 * @method getNoByIdByParams(array $params)
 * @method getProductsInOrderByParams(array $params)
 * @method getProductsInOrdersByParams(array $params)
 * @method getStaffRetailEffectByParams(array $params)
 * @method batchUpdateInvoiceIdByParams(array $params)
 * @method listInOrdersByParams(array $params)
 * @method getOrderAmountsPayableByParams(array $params)
 */
class Order {
    private static $instance;

    /**
     * 获取一个远程调用类
     * @return  static
     */
    public static function getDirectClass () {
        if (self::$instance == null) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    /**
     * 创建订单
     *
     * @support  int $store_id 门店ID [ require true ]
     * @support  int $himo_order_id 海马体订单id 
     * @support  int $user_id 下单用户ID 
     * @support  string $user_name 用户名 
     * @support  string $user_phone 用户手机号 
     * @support  string $user_sex 用户性别 
     * @support  string $user_birthday 用户生日 
     * @support  float $money 订单金额(商品原价总额) [Original type double] [ require true ]
     * @support  float $original_money 订单的原价总额 [Original type double] [ require true ]
     * @support  float $real_money 订单实付金额(商品优惠之后的总额) [Original type double] [ require true ]
     * @support  array $detail 订单详情 [ require true ]
     * @support  string $note 备注 
     * @support  int $input_staff_id 售出伙伴ID 
     * @throws  \Exception
     * @return  mixed
     */
    public static function create () {
        return self::getClass()->create();
    }

    /**
     * 编辑订单
     *
     * @support  string $order_no 订单号 
     * @support  int $id 订单ID 
     * @support  int $store_id 门店ID [ require true ]
     * @support  string $user_name 用户名 
     * @support  string $user_phone 用户手机号 
     * @support  string $user_sex 用户性别 
     * @support  string $user_birthday 用户生日 
     * @support  array $detail 订单详情 [ require true ]
     * @support  float $money 订单金额(商品原价总额) [Original type double] [ require true ]
     * @support  float $original_money 订单的原价总额 [Original type double] [ require true ]
     * @support  float $real_money 订单实付金额(商品优惠之后的总额) [Original type double] [ require true ]
     * @support  string $note 备注 
     * @support  int $input_staff_id 售出伙伴ID 
     * @throws  \Exception
     * @return  mixed
     */
    public static function update () {
        return self::getClass()->update();
    }

    /**
     * 订单详情
     *
     * @param  int $id 订单ID [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function detail (int $id) {
        return self::getClass()->detail([
            "id" => $id,
        ]);
    }

    /**
     * 订单详情【根据订单号获取】
     *
     * @param  string $order_no 订单号 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function detailByOrderNo (string $order_no) {
        return self::getClass()->detailByOrderNo([
            "order_no" => $order_no,
        ]);
    }

    /**
     * 提交订单
     *
     * @param  int $id 订单ID [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function submit (int $id) {
        return self::getClass()->submit([
            "id" => $id,
        ]);
    }

    /**
     * 删除订单
     *
     * @param  int $id 订单ID [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function remove (int $id) {
        return self::getClass()->remove([
            "id" => $id,
        ]);
    }

    /**
     * 删除订单（已支付）
     *
     * @param  int $id 订单ID [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function deletePaid (int $id) {
        return self::getClass()->deletePaid([
            "id" => $id,
        ]);
    }

    /**
     * 修改订单支付方式
     *
     * @param  int $id 订单ID [ require true ]
     * @param  string $pay_type 支付方式 [ require true ]
     * @param  string $payment_no 支付流水号 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function changePayType (int $id, string $pay_type, string $payment_no = '') {
        return self::getClass()->changePayType([
            "id" => $id,
            "pay_type" => $pay_type,
            "payment_no" => $payment_no,
        ]);
    }

    /**
     * 订单列表
     *
     * @support  int $page 页数 
     * @support  int $page_size 页内条数 
     * @support  string $user_phone 电话号码 
     * @support  string $paid_time_start 支付起始时间 
     * @support  string $paid_time_end 支付结束时间 
     * @support  string $user_name 用户姓名 
     * @support  string $order_no 订单号 
     * @support  int $store_id 门店ID 
     * @support  string $status 状态 
     * @support  array $status_list 状态列表 
     * @support  string $pay_type 支付方式 
     * @throws  \Exception
     * @return  mixed
     */
    public static function list () {
        return self::getClass()->list();
    }

    /**
     * 设置订单完成支付
     *
     * @param  int $id 订单ID [ require true ]
     * @param  float $money 订单金额 [Original type double] [ require true ]
     * @param  string $pay_type 支付类型 [ require true ]
     * @param  int $input_staff_id 售出伙伴ID [ require true ]
     * @param  string $payment_no 支付流水号 [ require true ]
     * @param  string $certificate 支付凭证 [ require true ]
     * @param  string $payment_time 对公打款时间 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function setPaid (int $id, float $money, string $pay_type, int $input_staff_id = 0, string $payment_no = '', string $certificate = '', string $payment_time = '') {
        return self::getClass()->setPaid([
            "id" => $id,
            "money" => $money,
            "pay_type" => $pay_type,
            "input_staff_id" => $input_staff_id,
            "payment_no" => $payment_no,
            "certificate" => $certificate,
            "payment_time" => $payment_time,
        ]);
    }

    /**
     * 关闭订单
     *
     * @param  int $id 订单ID [ require true ]
     * @param  string $reason 关闭原因 [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function close (int $id, string $reason) {
        return self::getClass()->close([
            "id" => $id,
            "reason" => $reason,
        ]);
    }

    /**
     * 修改用户数据
     *
     * @support  int $id 订单ID [ require true ]
     * @support  string $user_name 用户姓名 [ require true ]
     * @support  string $user_phone 用户手机号 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function changeUserData () {
        return self::getClass()->changeUserData();
    }

    /**
     * 根据订单号获取订单ID
     *
     * @param  string $order_no 订单号 [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function getIdByNo (string $order_no) {
        return self::getClass()->getIdByNo([
            "order_no" => $order_no,
        ]);
    }

    /**
     * 根据订单ID获取订单号
     *
     * @param  int $id 订单ID [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function getNoById (int $id) {
        return self::getClass()->getNoById([
            "id" => $id,
        ]);
    }

    /**
     * 根据订单号获取订单下面买的产品
     *
     * @param  string $order_no 订单号 [ require true ]
     * @param  int $order_id 订单ID [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function getProductsInOrder (string $order_no = NULL, int $order_id = NULL) {
        return self::getClass()->getProductsInOrder([
            "order_no" => $order_no,
            "order_id" => $order_id,
        ]);
    }

    /**
     * 根据订单号批量获取订单下面买的产品
     *
     * @param  array $order_nos 订单号数组 [ require true ]
     * @param  array $order_ids 订单ID数组 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function getProductsInOrders (array $order_nos = [], array $order_ids = []) {
        return self::getClass()->getProductsInOrders([
            "order_nos" => $order_nos,
            "order_ids" => $order_ids,
        ]);
    }

    /**
     * 获取伙伴时间范围内累积零售产品销售
     *
     * @param  int $staff_id 伙伴工号 [ require true ]
     * @param  string $start_time 开始时间 [ require true ]
     * @param  string $end_time 结束时间 [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function getStaffRetailEffect (int $staff_id, string $start_time, string $end_time) {
        return self::getClass()->getStaffRetailEffect([
            "staff_id" => $staff_id,
            "start_time" => $start_time,
            "end_time" => $end_time,
        ]);
    }

    /**
     * 批量更新订单的发票ID
     *
     * @param  int $invoice_id 发票ID [ require true ]
     * @param  array $order_nos 订单号数组 [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function batchUpdateInvoiceId (int $invoice_id, array $order_nos) {
        return self::getClass()->batchUpdateInvoiceId([
            "invoice_id" => $invoice_id,
            "order_nos" => $order_nos,
        ]);
    }

    /**
     * 批量获取订单列表
     *
     * @param  array $himo_order_ids 海马体订单ID数组 [ require true ]
     * @param  array $order_nos 订单号数组 [ require true ]
     * @param  array $ids 订单ID数组 [ require true ]
     * @param  array $order_user_ids 用户id数组 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function listInOrders (array $order_nos = [], array $ids = [], array $himo_order_ids = [], array $order_user_ids = []) {
        return self::getClass()->listInOrders([
            "order_nos" => $order_nos,
            "ids" => $ids,
            "himo_order_ids" => $himo_order_ids,
            "order_user_ids" => $order_user_ids,
        ]);
    }

    /**
     * 获取订单应付金额
     *
     * @param  int $id 订单id [ require true ]
     * @param  int $himo_order_id 海马体订单id [ require true ]
     * @throws  \Exception
     * @return  int
     */
    public static function getOrderAmountsPayable (int $id = 0, int $himo_order_id = 0) {
        return self::getClass()->getOrderAmountsPayable([
            "id" => $id,
            "himo_order_id" => $himo_order_id,
        ]);
    }

    /**
     * getClass
     * @deprecated
     * @throws  \Exception
     * @return  mixed
     */
    public static function getClass () {
        return RpcClass::getClass('Order', 'Retail\Order');
    }

    /**
     * registerMock
     * @throws  \Exception
     * @return  RpcMockClass
     */
    public static function registerMock () {
        return RpcClass::registerMockClass('Order', 'Retail\Order');
    }

    /**
     * @param  $name
     * @param  $arguments
     * @return  mixed
     * @throws  \Exception
     */
    public function __call ($name, $arguments) {
        if (substr($name, -8, 8) === 'ByParams') {
            $shortName = str_replace('ByParams', '', $name);
            return self::getClass()->$shortName(...$arguments);
        }

        throw new InvalidArgumentException('Method not found! : '.$name);
    }
}