<?php
namespace Mainto\Bridge\Invokes\Order\Order;

use Mainto\RpcServer\RpcClient\RpcClass;
use Mainto\RpcServer\RpcClient\RpcMockClass;
use InvalidArgumentException;

/**
 * Class Unique
 * @method refundListByParams(array $params)
 * @method refundDetailByParams(array $params)
 * @method getOrderRefundListByParams(array $params)
 * @method getOrderCurrentRefundStatusByParams(array $params)
 * @method getOrderMoneyDetailsByParams(array $params)
 * @method getOrderAlreadyRefundMoneyDetailsByParams(array $params)
 * @method getOrderCurrentCanRefundMoneyDetailsByParams(array $params)
 * @method getOrderOriginRefundDetailsByParams(array $params)
 * @method userCreateRefundByParams(array $params)
 * @method createRefundByParams(array $params)
 * @method rejectRefundByParams(array $params)
 * @method processRefundByParams(array $params)
 * @method createOtherRefundByParams(array $params)
 * @method processOtherRefundByParams(array $params)
 */
class Refund {
    private static $instance;

    /**
     * 获取一个远程调用类
     * @return  static
     */
    public static function getDirectClass () {
        if (self::$instance == null) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    /**
     * 查询后台退款列表
     *
     * @support  int $timeType 时间筛选类型 1：按照提交时间 2:按照退款时间 
     * @support  string $startTime 开始时间 
     * @support  string $endTime 结束时间 
     * @support  int $storeId 门店id 
     * @support  int $refundReason 退款理由 
     * @support  int $orderId 订单id 
     * @support  string $refundStatus 退款状态 
     * @support  string $handStatus 手动处理状态 
     * @support  string $refundType 退款类型 
     * @support  int $isStaffPay 是否伙伴垫付 1：是 
     * @support  int $page 页码 
     * @support  int $pageSize 条数 
     * @throws  \Exception
     * @return  array
     */
    public static function refundList () {
        return self::getClass()->refundList();
    }

    /**
     * 订单退款详情
     *
     * @param  int $refundId 退款id [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function refundDetail (int $refundId) {
        return self::getClass()->refundDetail([
            "refundId" => $refundId,
        ]);
    }

    /**
     * 根据订单id获取退款列表
     *
     * @param  int $orderId 订单id [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function getOrderRefundList (int $orderId) {
        return self::getClass()->getOrderRefundList([
            "orderId" => $orderId,
        ]);
    }

    /**
     * 根据订单id获取该订单当前退款状态
     * 退款状态包含以下信息：
     * 待处理退款数、退款成功数、退款拒绝数
     * 退款成功金额（订单部分、二销部分）
     * 退款处理中金额（订单部分、二销部分）
     * 当前剩余可退金额（订单部分、二销部分）
     *
     * @param  int $orderId 订单id [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function getOrderCurrentRefundStatus (int $orderId) {
        return self::getClass()->getOrderCurrentRefundStatus([
            "orderId" => $orderId,
        ]);
    }

    /**
     * 根据订单id获取订单金额详情
     * TODO 这个接口应该放到order controller里，暂时先放在这里
     *
     * @param  int $orderId 订单id [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function getOrderMoneyDetails (int $orderId) {
        return self::getClass()->getOrderMoneyDetails([
            "orderId" => $orderId,
        ]);
    }

    /**
     * 根据订单id获取该订单已退金额。处理中和退款成功的，都算已退
     *
     * @param  int $orderId 订单id [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function getOrderAlreadyRefundMoneyDetails (int $orderId) {
        return self::getClass()->getOrderAlreadyRefundMoneyDetails([
            "orderId" => $orderId,
        ]);
    }

    /**
     * 根据订单id获取该订单当前可退金额
     *
     * @param  int $orderId 订单id [ require true ]
     * @param  bool $includeGc 是否包含礼品卡，默认包含 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function getOrderCurrentCanRefundMoneyDetails (int $orderId, bool $includeGc = true) {
        return self::getClass()->getOrderCurrentCanRefundMoneyDetails([
            "orderId" => $orderId,
            "includeGc" => $includeGc,
        ]);
    }

    /**
     * 根据订单id获取该订单原始退款详情。只要订单无变化，那么无论订单是否发生过退款，原始退款详情都是不会变化的。
     *
     * @param  int $orderId 订单id [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function getOrderOriginRefundDetails (int $orderId) {
        return self::getClass()->getOrderOriginRefundDetails([
            "orderId" => $orderId,
        ]);
    }

    /**
     * 预约端用户自主申请退款
     *
     * @param  int $orderId 订单id [ require true ]
     * @param  int $reasonId 退款原因 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function userCreateRefund (int $orderId, int $reasonId) {
        return self::getClass()->userCreateRefund([
            "orderId" => $orderId,
            "reasonId" => $reasonId,
        ]);
    }

    /**
     * 管理端退款录入
     * 注意：管理端暂时禁用掉退款到礼品卡（refund_to_gift_card）这种类型，因为这种类型不利于后期按产品退款的需求，会导致数据不准确
     *
     * @support  int $orderId 订单ID [ require true ]
     * @support  int $refundStoreId 退款门店ID 
     * @support  string $refundType 退款类型 [ require true ]
     * @support  string $refundPart 退款部分，当选择部分退款时需要提供 
     * @support  float $refundMoney 退款金额，当选择部分退款时需要提供 [Original type double] [ require true ]
     * @support  string $refundPeople 退款人 [ require true ]
     * @support  string $refundAccount 退款账户 [ require true ]
     * @support  int $refundStaffId 退款录入人ID [ require true ]
     * @support  int $refundScale 退款比例 [ require true ]
     * @support  int $refundReason 退款原因 [ require true ]
     * @support  bool $isStaffPay 是否为伙伴垫付 
     * @support  bool $isRefundCard 是否退券 
     * @support  string $refundNote 退款备注 
     * @throws  \Exception
     * @return  array
     */
    public static function createRefund () {
        return self::getClass()->createRefund();
    }

    /**
     * 管理端拒绝退款
     *
     * @param  int $refundId 退款id [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function rejectRefund (int $refundId) {
        return self::getClass()->rejectRefund([
            "refundId" => $refundId,
        ]);
    }

    /**
     * 管理端处理退款（行为相当于同意退款）
     *
     * @param  int $refundId 退款id [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function processRefund (int $refundId) {
        return self::getClass()->processRefund([
            "refundId" => $refundId,
        ]);
    }

    /**
     * 其它退款录入
     *
     * @support  int $refundStoreId 退款门店ID [ require true ]
     * @support  float $refundMoney 退款金额 [Original type double] [ require true ]
     * @support  int $refundGoodsId 其他退款类型需要的商品ID 
     * @support  string $refundPeople 退款人 [ require true ]
     * @support  string $refundAccount 退款账户 [ require true ]
     * @support  int $refundStaffId 退款录入人ID [ require true ]
     * @support  int $refundScale 退款比例 [ require true ]
     * @support  int $refundReason 退款原因 [ require true ]
     * @support  bool $isStaffPay 是否为伙伴垫付 
     * @support  string $refundNote 退款备注 
     * @throws  \Exception
     * @return  array
     */
    public static function createOtherRefund () {
        return self::getClass()->createOtherRefund();
    }

    /**
     * 处理其它退款（行为相当于同意退款）
     *
     * @param  int $refundId 退款id [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function processOtherRefund (int $refundId) {
        return self::getClass()->processOtherRefund([
            "refundId" => $refundId,
        ]);
    }

    /**
     * getClass
     * @deprecated
     * @throws  \Exception
     * @return  mixed
     */
    public static function getClass () {
        return RpcClass::getClass('Order', 'Refund');
    }

    /**
     * registerMock
     * @throws  \Exception
     * @return  RpcMockClass
     */
    public static function registerMock () {
        return RpcClass::registerMockClass('Order', 'Refund');
    }

    /**
     * @param  $name
     * @param  $arguments
     * @return  mixed
     * @throws  \Exception
     */
    public function __call ($name, $arguments) {
        if (substr($name, -8, 8) === 'ByParams') {
            $shortName = str_replace('ByParams', '', $name);
            return self::getClass()->$shortName(...$arguments);
        }

        throw new InvalidArgumentException('Method not found! : '.$name);
    }
}