<?php
namespace Mainto\Bridge\Invokes\ManageAuth;

use Mainto\RpcServer\RpcClient\RpcClass;
use Mainto\RpcServer\RpcClient\RpcMockClass;
use Mainto\RpcServer\RpcServer\RpcDefinition;
use InvalidArgumentException;

/**
 * Class Unique
 * @method getStaffRoleListByParams(array $params)
 * @method getStaffRoleListByPageByParams(array $params)
 * @method getRelationIdByParams(array $params)
 * @method getStaffPermissionListByParams(array $params)
 * @method getStaffPermissionListByPageByParams(array $params)
 * @method grantRoleByParams(array $params)
 * @method grantPermissionByParams(array $params)
 * @method grantBatchPermissionByParams(array $params)
 * @method revokeRoleByParams(array $params)
 * @method revokePermissionByParams(array $params)
 * @method getAdminByStoreByParams(array $params)
 * @method getAdminIdByStoreByParams(array $params)
 * @method getAdminIdByStoreIdsByParams(array $params)
 * @method getAdminByAppIdByParams(array $params)
 * @method deleteStaffAppPermissionByParams(array $params)
 * @method handleStaffAppDataByParams(array $params)
 * @method getStaffByPermissionByParams(array $params)
 * @method getStaffsByRoleByParams(array $params)
 * @method getStaffRoleListsByParams(array $params)
 * @method getStaffIdsContainPermissionsByParams(array $params)
 */
class Staff {
    private static $instance;

    /**
     * 获取一个远程调用类
     * @return  static
     */
    public static function getDirectClass () {
        if (self::$instance == null) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    /**
     * 获取伙伴已授权角色
     *
     * @param  int $staffId 伙伴ID [ require true ]
     * @param  int $appId application Id [ require false ]
     * @throws  \Exception
     * @return  \Mainto\Bridge\Structs\ManageAuth\StaffRoleDTO[]
     */
    public static function getStaffRoleList (int $staffId, int $appId = 0) {
        return app('json_mapper')->mapUseTypeString(
            self::getClass()->getStaffRoleList([
                            "staffId" => $staffId,
                            "appId" => $appId,
                        ]),
            "\Mainto\Bridge\Structs\ManageAuth\StaffRoleDTO[]"
        );
    }

    /**
     * 获取伙伴已授权角色，分页
     *
     * @support  int $staffId 伙伴ID [ require false ]
     * @support  array $staffIds 伙伴IDs [ require false ]
     * @support  bool $withRoleInfo 包含角色权限 [ require false ]
     * @support  bool $withRoleContainPermission 角色包含的权限详情 [ require false ]
     * @param  int $page 分页 [ require false ]
     * @param  int $pageSize 页码 [ require false ]
     * @throws  \Exception
     * @return  \Mainto\Bridge\Structs\ManageAuth\StaffRolePageDTO
     */
    public static function getStaffRoleListByPage (int $page = 1, int $pageSize = 10) {
        return app('json_mapper')->mapUseTypeString(
            self::getClass()->getStaffRoleListByPage([
                            "page" => $page,
                            "pageSize" => $pageSize,
                        ]),
            "\Mainto\Bridge\Structs\ManageAuth\StaffRolePageDTO"
        );
    }

    /**
     * 获取relation id，接口不规范，后续更改
     *
     * @param  int $staffId 员工id [ require true ]
     * @param  int $keyId 关键值id(role_id 或 permission_id) [ require true ]
     * @param  int $type 类型 (1 staffRole 2: staffPermission) [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function getRelationId (int $staffId, int $keyId, int $type) {
        return self::getClass()->getRelationId([
            "staffId" => $staffId,
            "keyId" => $keyId,
            "type" => $type,
        ]);
    }

    /**
     * 获取伙伴已授权权限
     *
     * @param  int $staff_id 伙伴ID [ require true ]
     * @param  bool $isRole 是否需要role权限 [ require false ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function getStaffPermissionList (int $staff_id, bool $isRole = true) {
        return self::getClass()->getStaffPermissionList([
            "staff_id" => $staff_id,
            "isRole" => $isRole,
        ]);
    }

    /**
     * 获取伙伴已授权权限，分页
     *
     * @support  int $staffId 伙伴ID [ require false ]
     * @support  array $staffIds 伙伴IDs [ require false ]
     * @support  bool $withPermission 包含permission详情 [ require false ]
     * @param  int $page 分页 [ require false ]
     * @param  int $pageSize 页码 [ require false ]
     * @throws  \Exception
     * @return  \Mainto\Bridge\Structs\ManageAuth\StaffPermissionPageDTO
     */
    public static function getStaffPermissionListByPage (int $page = 1, int $pageSize = 10) {
        return app('json_mapper')->mapUseTypeString(
            self::getClass()->getStaffPermissionListByPage([
                            "page" => $page,
                            "pageSize" => $pageSize,
                        ]),
            "\Mainto\Bridge\Structs\ManageAuth\StaffPermissionPageDTO"
        );
    }

    /**
     * 授予伙伴角色
     *
     * @param  int $roleId 角色ID [ require true ]
     * @param  int $staffId 伙伴ID [ require true ]
     * @param  int $effectAt 起效时间(0-立即生效) [ require false ]
     * @param  int $expireAt 过期时间(0-永不过期) [ require false ]
     * @throws  \Exception
     * @return  int
     */
    public static function grantRole (int $roleId, int $staffId, int $effectAt = 0, int $expireAt = 0) {
        return self::getClass()->grantRole([
            "roleId" => $roleId,
            "staffId" => $staffId,
            "effectAt" => $effectAt,
            "expireAt" => $expireAt,
        ]);
    }

    /**
     * 授予伙伴权限
     *
     * @param  int $permissionId 权限ID [ require true ]
     * @param  int $staffId 伙伴ID [ require true ]
     * @param  int $effectAt 起效时间(0-立即生效) [ require false ]
     * @param  int $expireAt 过期时间(0-永不过期) [ require false ]
     * @throws  \Exception
     * @return  int
     */
    public static function grantPermission (int $permissionId, int $staffId, int $effectAt = 0, int $expireAt = 0) {
        return self::getClass()->grantPermission([
            "permissionId" => $permissionId,
            "staffId" => $staffId,
            "effectAt" => $effectAt,
            "expireAt" => $expireAt,
        ]);
    }

    /**
     * 批量授予伙伴权限
     *
     * @param  array $addPermissionIds 需要增加权限列表 [ require false ]
     * @param  array $delPermissionIds 需要删除权限列表 [ require false ]
     * @param  int $staff_id 伙伴id [ require true ]
     * @param  int $effectAt 起效时间(0-立即生效) [ require false ]
     * @param  int $expireAt 过期时间(0-永不过期) [ require false ]
     * @throws  \Exception
     * @return  bool
     */
    public static function grantBatchPermission (int $staff_id, array $addPermissionIds = [], array $delPermissionIds = [], int $effectAt = 0, int $expireAt = 0) {
        return self::getClass()->grantBatchPermission([
            "staff_id" => $staff_id,
            "addPermissionIds" => $addPermissionIds,
            "delPermissionIds" => $delPermissionIds,
            "effectAt" => $effectAt,
            "expireAt" => $expireAt,
        ]);
    }

    /**
     * 撤销伙伴角色
     *
     * @param  int $relationId 授权关系ID [ require true ]
     * @throws  \Exception
     * @return  bool
     */
    public static function revokeRole (int $relationId) {
        return self::getClass()->revokeRole([
            "relationId" => $relationId,
        ]);
    }

    /**
     * 撤销伙伴权限
     *
     * @param  int $relationId 授权关系ID [ require true ]
     * @throws  \Exception
     * @return  bool
     */
    public static function revokePermission (int $relationId) {
        return self::getClass()->revokePermission([
            "relationId" => $relationId,
        ]);
    }

    /**
     * 根据门店获取管理员列表
     *
     * @param  int $store_id 门店id [ require true ]
     * @param  array $staff_ids 伙伴id [ require false ]
     * @param  int $page 页码 [ require false ]
     * @param  int $page_size 条数 [ require false ]
     * @throws  \Exception
     * @return  \Mainto\Bridge\Structs\ManageAuth\StaffStorePageDTO
     */
    public static function getAdminByStore (int $store_id, array $staff_ids = [], int $page = 1, int $page_size = 15) {
        return app('json_mapper')->mapUseTypeString(
            self::getClass()->getAdminByStore([
                            "store_id" => $store_id,
                            "staff_ids" => $staff_ids,
                            "page" => $page,
                            "page_size" => $page_size,
                        ]),
            "\Mainto\Bridge\Structs\ManageAuth\StaffStorePageDTO"
        );
    }

    /**
     * 根据门店获取管理员ID数组
     *
     * @param  int $store_id 门店id [ require true ]
     * @throws  \Exception
     * @return  int[]
     */
    public static function getAdminIdByStore (int $store_id) {
        return self::getClass()->getAdminIdByStore([
            "store_id" => $store_id,
        ]);
    }

    /**
     * 根据门店ids获取管理员ID数组
     *
     * @param  array $storeIds 门店ids [ require true ]
     * @throws  \Exception
     * @return  \Mainto\Bridge\Structs\ManageAuth\StaffStoreDTO[]
     */
    public static function getAdminIdByStoreIds (array $storeIds) {
        return app('json_mapper')->mapUseTypeString(
            self::getClass()->getAdminIdByStoreIds([
                            "storeIds" => $storeIds,
                        ]),
            "\Mainto\Bridge\Structs\ManageAuth\StaffStoreDTO[]"
        );
    }

    /**
     * 根据application获取管理员列表
     *
     * @param  int $app_id application id [ require false ]
     * @param  array $staff_ids 员工id [ require false ]
     * @param  int $page 页数 [ require false ]
     * @param  int $page_size 条数 [ require false ]
     * @param  int $status 状态 [ require false ]
     * @throws  \Exception
     * @return  \Mainto\Bridge\Structs\ManageAuth\StaffApplicationPageDTO
     */
    public static function getAdminByAppId (int $app_id = 0, array $staff_ids = [], int $status = 0, int $page = 1, int $page_size = 15) {
        return app('json_mapper')->mapUseTypeString(
            self::getClass()->getAdminByAppId([
                            "app_id" => $app_id,
                            "staff_ids" => $staff_ids,
                            "status" => $status,
                            "page" => $page,
                            "page_size" => $page_size,
                        ]),
            "\Mainto\Bridge\Structs\ManageAuth\StaffApplicationPageDTO"
        );
    }

    /**
     * 删除伙伴应用数据及相关应用的权限
     *
     * @param  int $staffId 伙伴id [ require true ]
     * @param  int $appId 应用ID [ require true ]
     * @throws  \Exception
     * @return  bool
     */
    public static function deleteStaffAppPermission (int $staffId, int $appId) {
        return self::getClass()->deleteStaffAppPermission([
            "staffId" => $staffId,
            "appId" => $appId,
        ]);
    }

    /**
     * 创建staff application 数据
     *
     * @param  int $staffId 伙伴id [ require true ]
     * @param  int $appId 应用ID [ require true ]
     * @param  int $status 状态 1:开启 2关闭 [ require true ]
     * @throws  \Exception
     * @return  bool
     */
    public static function handleStaffAppData (int $staffId, int $appId, int $status) {
        return self::getClass()->handleStaffAppData([
            "staffId" => $staffId,
            "appId" => $appId,
            "status" => $status,
        ]);
    }

    /**
     * 根据权限id获取对应伙伴
     *
     * @param  int $permissionId 权限id [ require true ]
     * @throws  \Exception
     * @return  int[]
     */
    public static function getStaffByPermission (int $permissionId) {
        return self::getClass()->getStaffByPermission([
            "permissionId" => $permissionId,
        ]);
    }

    /**
     * 根据角色获取对应的伙伴
     *
     * @param  int $roleId 角色id [ require true ]
     * @throws  \Exception
     * @return  \Mainto\Bridge\Structs\ManageAuth\StaffRoleDTO[]
     */
    public static function getStaffsByRole (int $roleId) {
        return app('json_mapper')->mapUseTypeString(
            self::getClass()->getStaffsByRole([
                            "roleId" => $roleId,
                        ]),
            "\Mainto\Bridge\Structs\ManageAuth\StaffRoleDTO[]"
        );
    }

    /**
     * 批量获取伙伴已授权角色
     *
     * @param  array $staffIds 伙伴ID [ require true ]
     * @param  int $appId application Id [ require false ]
     * @throws  \Exception
     * @return  \Mainto\Bridge\Structs\ManageAuth\StaffRoleDTO[]
     */
    public static function getStaffRoleLists (array $staffIds, int $appId = 0) {
        return app('json_mapper')->mapUseTypeString(
            self::getClass()->getStaffRoleLists([
                            "staffIds" => $staffIds,
                            "appId" => $appId,
                        ]),
            "\Mainto\Bridge\Structs\ManageAuth\StaffRoleDTO[]"
        );
    }

    /**
     * 获取拥有指定权限伙伴IDs
     *
     * @param  array $staffIds 伙伴IDs [ require true ]
     * @param  string $containPermissionName 权限名称 [ require true ]
     * @throws  \Exception
     * @return  int[]
     */
    public static function getStaffIdsContainPermissions (array $staffIds, string $containPermissionName) {
        return self::getClass()->getStaffIdsContainPermissions([
            "staffIds" => $staffIds,
            "containPermissionName" => $containPermissionName,
        ]);
    }

    /**
     * getClass
     * @deprecated
     * @throws  \Exception
     * @return  mixed
     */
    public static function getClass () {
        return RpcClass::getClass('ManageAuth', 'Staff');
    }

    /**
     * registerMock
     * @throws  \Exception
     * @return  RpcMockClass
     */
    public static function registerMock () {
        return RpcClass::registerMockClass('ManageAuth', 'Staff');
    }

    /**
     * @param  $name
     * @param  $arguments
     * @return  mixed
     * @throws  \Exception
     */
    public function __call ($name, $arguments) {
        if (substr($name, -8, 8) === 'ByParams') {
            $shortName = str_replace('ByParams', '', $name);
            return self::getClass()->$shortName(...$arguments);
        }

        throw new InvalidArgumentException('Method not found! : '.$name);
    }
}