<?php
namespace Mainto\Bridge\Invokes\HimoTraining;

use Mainto\RpcServer\RpcClient\RpcClass;
use Mainto\RpcServer\RpcClient\RpcMockClass;
use InvalidArgumentException;

/**
 * Class Unique
 * @method createByParams(array $params)
 * @method releaseByParams(array $params)
 * @method unReleaseByParams(array $params)
 * @method updateOrNewByParams(array $params)
 * @method updateUnreleaseByParams(array $params)
 * @method getInfoByParams(array $params)
 * @method getListByParams(array $params)
 * @method getListByPageByParams(array $params)
 * @method getCourseCountByGroupIdsByParams(array $params)
 * @method deleteByParams(array $params)
 * @method getNextCourseIdsByParams(array $params)
 */
class Course {
    private static $instance;

    /**
     * 获取一个远程调用类
     * @return  static
     */
    public static function getDirectClass () {
        if (self::$instance == null) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    /**
     * 创建课程
     *
     * @param  int $category_id 课程分类id [ require true ]
     * @support  string $name 课程名称 [ require true ]
     * @support  array $label_ids 标签id数组 [Original type array&lt;uint&gt;] 
     * @support  string $description 课程描述 
     * @support  int $pass_credit 课程学分 [Original type uint] 
     * @support  string $image_path 课程封面图 [ require true ]
     * @param  int $update_by 更新人员 [ require true ]
     * @param  array $can_rule 前置条件 [ require true ]
     * @support  array $children 子课程配置 
     * @support  array $sort_rule 排序规则 
     * @param  array $groups 适用分组 [Original type array&lt;int&gt;] [ require true ]
     * @support  array $learn_data 学习资料 
     * @support  int $teacher_id 讲师id [Original type uint] 
     * @throws  \Exception
     * @return  int
     */
    public static function create (int $category_id, int $update_by, array $groups, array $can_rule = []) {
        return self::getClass()->create([
            "category_id" => $category_id,
            "update_by" => $update_by,
            "groups" => $groups,
            "can_rule" => $can_rule,
        ]);
    }

    /**
     * 上线
     *
     * @param  int $id 课程id [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function release (int $id) {
        return self::getClass()->release([
            "id" => $id,
        ]);
    }

    /**
     * 下线
     *
     * @param  int $id 课程id [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function unRelease (int $id) {
        return self::getClass()->unRelease([
            "id" => $id,
        ]);
    }

    /**
     * 更新课程
     *
     * @param  int $id 课程id [ require true ]
     * @support  int $pass_credit 课程学分 [Original type uint] 
     * @support  int $category_id 课程分类id 
     * @support  string $name 课程名称 
     * @support  array $label_ids 标签id数组 [Original type array&lt;uint&gt;] 
     * @support  string $description 课程描述 
     * @support  string $image_path 课程封面图 
     * @param  int $update_by 更新人员 [ require true ]
     * @support  array $can_rule 前置条件 
     * @support  array $sort_rule 排序规则 
     * @support  array $children 子课程配置 
     * @param  array $groups 适用分组 [Original type array&lt;int&gt;] [ require true ]
     * @support  array $learn_data 学习资料 
     * @support  int $teacher_id 讲师id [Original type uint] 
     * @throws  \Exception
     * @return  mixed
     */
    public static function updateOrNew (int $id, int $update_by, array $groups) {
        return self::getClass()->updateOrNew([
            "id" => $id,
            "update_by" => $update_by,
            "groups" => $groups,
        ]);
    }

    /**
     * 更新线上课程
     *
     * @param  int $id 课程id [ require true ]
     * @support  int $category_id 课程分类id 
     * @support  string $name 课程名称 
     * @support  array $label_ids 标签id数组 [Original type array&lt;uint&gt;] 
     * @support  string $description 课程描述 
     * @support  string $image_path 课程封面图 
     * @param  int $update_by 更新人员 [ require true ]
     * @support  array $can_rule 前置条件 
     * @support  array $sort_rule 排序规则 
     * @param  array $groups 适用分组 [Original type array&lt;int&gt;] [ require true ]
     * @support  array $learn_data 学习资料 
     * @support  int $teacher_id 讲师id [Original type uint] 
     * @throws  \Exception
     * @return  mixed
     */
    public static function updateUnrelease (int $id, int $update_by, array $groups) {
        return self::getClass()->updateUnrelease([
            "id" => $id,
            "update_by" => $update_by,
            "groups" => $groups,
        ]);
    }

    /**
     * 获取课程详情
     *
     * @param  int $id 课程id [ require true ]
     * @support  bool $with_category 包含分类信息 
     * @support  bool $with_groups 包含分组信息 
     * @support  bool $with_labels 包含标签信息 
     * @support  bool $with_learnData 包含课程学习资料信息 
     * @support  bool $with_teacher 包含讲师信息 
     * @support  bool $with_children 包含子课程信息 
     * @support  bool $with_exam 包含课程测试信息 
     * @support  bool $with_schedule 包含课程测试信息 
     * @throws  \Exception
     * @return  mixed
     */
    public static function getInfo (int $id) {
        return self::getClass()->getInfo([
            "id" => $id,
        ]);
    }

    /**
     * 条件查询课程
     *
     * @support  array $ids 课程ids [Original type array&lt;uint&gt;] 
     * @support  string $name 课程名词模糊查询 
     * @support  string $group_by 分组字段 
     * @support  array $category_ids 课程分类ids [Original type array&lt;int&gt;] 
     * @support  array $group_ids 员工分组ids [Original type array&lt;int&gt;] 
     * @support  int $is_released 是否已上线 
     * @support  bool $with_learnData 包含课程学习资料信息 
     * @support  bool $with_groups_info 包含分组信息 
     * @support  bool $with_schedule 包含课程测试信息 
     * @support  bool $has_exam 关联练习 
     * @throws  \Exception
     * @return  array
     */
    public static function getList () {
        return self::getClass()->getList();
    }

    /**
     * 课程列表分页
     *
     * @support  array $ids 课程ids 
     * @support  string $name 课程名称 
     * @support  array $category_ids 课程分类ids [Original type array&lt;int&gt;] 
     * @support  array $label_ids 标签ids [Original type array&lt;uint&gt;] 
     * @support  int $is_released 是否上线 
     * @support  array $group_ids 员工分组ids [Original type array&lt;int&gt;] 
     * @support  array $order_by 排序[[&#039;column&#039;, &#039;desc|asc&#039;]] 
     * @support  bool $with_learnData 包含课程学习资料信息 
     * @support  bool $with_groups 包含分组信息 
     * @support  bool $with_children 包含子课程信息 
     * @support  bool $with_schedule 包含课程测试信息 
     * @support  bool $with_exam 是否包含试卷 
     * @param  int $page 页码 [ require true ]
     * @param  int $pageSize 每页条数 [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function getListByPage (int $page = 1, int $pageSize = 10) {
        return self::getClass()->getListByPage([
            "page" => $page,
            "pageSize" => $pageSize,
        ]);
    }

    /**
     * 根据分组id数组获取对应分组关联的课程的数目
     *
     * @param  array $groupIds 分组id数组 [Original type array&lt;uint&gt;] [ require true ]
     * @throws  \Exception
     * @return  array
     */
    public static function getCourseCountByGroupIds (array $groupIds) {
        return self::getClass()->getCourseCountByGroupIds([
            "groupIds" => $groupIds,
        ]);
    }

    /**
     * 删除课程
     *
     * @param  int $id 课程id [Original type uint] [ require true ]
     * @throws  \Exception
     * @return  mixed
     */
    public static function delete (int $id) {
        return self::getClass()->delete([
            "id" => $id,
        ]);
    }

    /**
     * 获取课程依赖id数组
     *
     * @param  int $course_id 课程id [Original type uint] [ require true ]
     * @support  array $group_ids 课程分组 [Original type array&lt;uint&gt;] 
     * @throws  \Exception
     * @return  array
     */
    public static function getNextCourseIds (int $course_id) {
        return self::getClass()->getNextCourseIds([
            "course_id" => $course_id,
        ]);
    }

    /**
     * getClass
     * @deprecated
     * @throws  \Exception
     * @return  mixed
     */
    public static function getClass () {
        return RpcClass::getClass('HimoTraining', 'Course');
    }

    /**
     * registerMock
     * @throws  \Exception
     * @return  RpcMockClass
     */
    public static function registerMock () {
        return RpcClass::registerMockClass('HimoTraining', 'Course');
    }

    /**
     * @param  $name
     * @param  $arguments
     * @return  mixed
     * @throws  \Exception
     */
    public function __call ($name, $arguments) {
        if (substr($name, -8, 8) === 'ByParams') {
            $shortName = str_replace('ByParams', '', $name);
            return self::getClass()->$shortName(...$arguments);
        }

        throw new InvalidArgumentException('Method not found! : '.$name);
    }
}