<?php

/**
 * Class Map
 * @package  Mainto\RpcServer\Util\Types
 * @template T
 */
class Map implements JsonSerializable, Iterator, ArrayAccess {
    /**
     * @var T[]
     */
    private array $items;

    /**
     * Map constructor.
     * @param T[] $items
     */
    public function __construct (array $items = []) {
        $this->items = $items;
    }

    /**
     * @param T[] $items
     */
    public function setItems (array $items): void {
        $this->items = $items;
    }

    /**
     * @param $key
     * @param T $value
     */
    public function set ($key, mixed $value): void {
        $this->items[$key] = $value;
    }

    /**
     * @param $key
     * @return T|null
     */
    public function get ($key): mixed {
        return $this->items[$key] ?? null;
    }

    public function toJson ($options = 0) {
        if ($this->items) {
            return json_encode($this->items, $options);
        } else {
            return '{}';
        }
    }

    public function jsonSerialize (): static|array {
        if ($this->items) {
            return $this->items;
        } else {
            return $this;
        }
    }

    public function offsetExists ($offset): bool {
        return array_key_exists($offset, $this->items);
    }

    /**
     * @param mixed $offset
     * @return mixed
     * @throws Exception
     */
    public function offsetGet ($offset): mixed {
        if (array_key_exists($offset, $this->items)) {
            return $this->items[$offset];
        }

        throw new Exception("Undefined offset: $offset");
    }

    public function offsetSet ($offset, $value): void {
        $this->items[$offset] = $value;
    }

    public function offsetUnset ($offset): void {
        unset($this->items[$offset]);
    }

    /**
     * @return T
     */
    public function current (): mixed {
        return current($this->items);
    }

    public function next (): void {
        next($this->items);
    }

    public function key (): string|int|null {
        return key($this->items);
    }

    public function valid (): bool {
        return key($this->items) !== null;
    }

    public function rewind (): void {
        reset($this->items);
    }

    public function filter (callable $callback): array {
        return array_filter($this->items, $callback);
    }

    public function map (callable $callback): array {
        $map = [];
        foreach ($this->items as $key => $value) {
            $map[] = $callback($value, $key);
        }

        return $map;
    }

    public function mapWithKey (callable $callback): array {
        return array_map_with_Key($callback, $this->items);
    }

    public function toArray (): array {
        return to_array_map($this->items);
    }
}