<?php


namespace Mainto\JsonMapper\Mapper;

use Mainto\JsonMapper\Exceptions\JsonMapperException;
use Map;
use ReflectionException;

/**
 * Class JsonMapper
 * @package Mainto\JsonMapper\Mapper
 */
class JsonMapper implements MapperInterface {
    public array $beforeSetter = [];
    public bool $ignoreUnknownProperties = true;
    protected array $simpleType = [
        'bool',
        'int',
        'string',
        'float',
        'array',
    ];
    /**
     * @var array ['class_key' => ClassRef]
     */
    private array $classRefMap = [];

    /**
     * @param $data
     * @param object $object
     */
    public function map ($data, object $object) {
        if (is_string($data)) {
            $data = json_decode($data, true);
        }

        $this->mapJsonArray(new Context($data), $object);
    }

    public function mapJsonArray (Context $context, object $object) {
        $classRef = $this->classRef($object);
        $className = $classRef->namespace.'\\'.$classRef->name;

        $data = $context->value();

        if (!$this->ignoreUnknownProperties) {
            $diff = array_diff(array_keys($data), array_map(function (PropertyRef $propertyRef) { return $propertyRef->serializableName; }, $classRef->properties));
            if ($diff) {
                throw new JsonMapperException(sprintf("unknown properties [%s] for $className", implode(', ', $diff)));
            }
        }

        foreach ($classRef->properties as $propertyRef) {
            $dataName = $propertyRef->serializableName;
            if (!array_key_exists($dataName, $data)) {
                continue;
            }

            $value = $this->mapUseTypeString(new Context($data[$dataName], $context), $propertyRef->type);

            if (isset($this->beforeSetter[$className])) {
                $this->beforeSetter[$className]($propertyRef->name, $value);
            }

            if ($propertyRef->setterMethod) {
                $object->{$propertyRef->setterMethod}($value);
            } else {
                $propertyRef->set($object, $value);
            }
        }
    }

    /**
     * @param object $object
     * @return ClassRef
     * @throws ReflectionException
     */
    public function classRef (object $object): ClassRef {
        $propertiesKey = get_class($object);

        return $this->classRefMap[$propertiesKey] ?? ($this->classRefMap[$propertiesKey] = new ClassRef($object));
    }

    /**
     * @param $data
     * @param string $typeString
     * @return array|Map|mixed
     */
    public function mapUseTypeString (Context $context, string $typeString) {
        $data = $context->value();
        if (is_subclass_of($typeString, DynamicType::class)) {
            $value = $this->initObject($typeString::dynamicType($context), $context);
        } else if (in_array($typeString, $this->simpleType) || $typeString == 'mixed' || $data === null) {
            $value = $data;
        } elseif (str_ends_with($typeString, '[]')) {
            $context->valueDecode();

            $value = $this->warpArray($typeString, $context);
        } elseif (str_starts_with($typeString, 'Map<')) {
            $value = new Map();
            $keyValueType = explode(',', substr($typeString, 4, -1));
            $subType = trim($keyValueType[1]);
            if (in_array($subType, $this->simpleType)) {
                $value->setItems($data);
            } else {
                foreach ($data as $key => $datum) {
                    $subObject = $this->initObject($subType, new Context($datum, $context));
                    $value->set($key, $subObject);
                }
            }
        } else {
            $value = $this->initObject($typeString, $context);
        }

        return $value;
    }

    private function initObject ($objectClass, Context $context) {
        if (is_subclass_of($objectClass, JsonUnserializable::class)) {
            return $objectClass::jsonUnserialize($context->value());
        } else {
            $propertyObject = new $objectClass;
            $this->mapWithContext($context, $propertyObject);

            return $propertyObject;
        }
    }

    public function mapWithContext (Context $context, object $object) {
        $context->valueDecode();

        $this->mapJsonArray($context, $object);
    }

    public function warpArray ($type, Context $context) {
        $data = $context->value();

        if (str_ends_with($type, '[]')) {
            $subType = substr($type, 0, -2);
            $value = [];
            foreach ($data as $key => $item) {
                $value[$key] = $this->warpArray($subType, new Context($item, $context));
            }
            return $value;
        } else {
            if (in_array($type, $this->simpleType) || $type == 'mixed') {
                return $data;
            } else {
                //                $subObject = $this->initObject($type, $context);

                //                $this->map($data, $subObject);
                return $this->initObject($type, $context);
            }
        }
    }
}