<?php


namespace Mainto\JsonMapper\Mapper;

use Mainto\JsonMapper\Exceptions\JsonMapperException;
use Map;
use ReflectionException;

/**
 * Class JsonMapper
 * @package Mainto\JsonMapper\Mapper
 */
class JsonMapper implements MapperInterface {
    /**
     * @var array ['class_key' => ClassRef]
     */
    private array $classRefMap = [];

    protected array $simpleType = [
        'bool',
        'int',
        'string',
        'float',
        'array',
    ];

    public array $beforeSetter = [];

    public bool $ignoreUnknownProperties = true;

    /**
     * @param $data
     * @param object $object
     */
    public function map ($data, object $object) {
        if (is_string($data)) {
            $data = json_decode($data, true);
        }

        $this->mapJsonArray($data, $object);
    }

    /**
     * @param $data
     * @param string $typeString
     * @return array|Map|mixed
     */
    public function mapUseTypeString ($data, string $typeString) {
        if (in_array($typeString, $this->simpleType) || $typeString == 'mixed' || $data === null) {
            $value = $data;
        } elseif (str_ends_with($typeString, '[]')) {
            if (is_string($data)) {
                $data = json_decode($data , true);
            }

            $value = $this->warpArray($typeString, $data);
        } elseif (str_starts_with($typeString, 'Map<')) {
            $value = new Map();
            $keyValueType = explode(',', substr($typeString, 4, -1));
            $subType = trim($keyValueType[1]);
            if (in_array($subType, $this->simpleType)) {
                $value->setItems($data);
            } else {
                foreach ($data as $key => $datum) {
                    $subObject = $this->initObject($subType, $datum);
                    $value->set($key, $subObject);
                }
            }
        } else {
            $value = $this->initObject($typeString, $data);
        }

        return $value;
    }

    private function initObject ($objectClass, $data) {
        if (is_subclass_of($objectClass, JsonUnserializable::class)) {
            return $objectClass::jsonUnserialize($data);
        } else {
            $propertyObject = new $objectClass;
            $this->map($data, $propertyObject);

            return $propertyObject;
        }
    }

    public function warpArray ($type, $data) {
        if (str_ends_with($type, '[]')) {
            $subType = substr($type, 0, -2);
            $value = [];
            foreach ($data as $key => $item) {
                $value[$key] = $this->warpArray($subType, $item);
            }
            return $value;
        } else {
            if (in_array($type, $this->simpleType) || $type == 'mixed') {
                return $data;
            } else {
                $subObject = $this->initObject($type, $data);

                $this->map($data, $subObject);
                return $subObject;
            }
        }
    }

    public function mapJsonArray (array $data, object $object) {
        $classRef = $this->classRef($object);
        $className = $classRef->namespace.'\\'.$classRef->name;

        if (!$this->ignoreUnknownProperties) {
            $diff = array_diff(array_keys($data), array_map(function (PropertyRef $propertyRef) { return $propertyRef->serializableName; }, $classRef->properties));
            if ($diff) {
                throw new JsonMapperException(sprintf("unknown properties [%s] for $className", implode(', ', $diff)));
            }
        }

        foreach ($classRef->properties as $propertyRef) {
            $dataName = $propertyRef->serializableName;
            if (!array_key_exists($dataName, $data)) {
                continue;
            }

            $value = $this->mapUseTypeString($data[$dataName], $propertyRef->type);

            if (isset($this->beforeSetter[$className])) {
                $this->beforeSetter[$className]($propertyRef->name, $value);
            }

            if ($propertyRef->setterMethod) {
                $object->{$propertyRef->setterMethod}($value);
            } else {
                $propertyRef->set($object, $value);
            }
        }
    }

    /**
     * @param object $object
     * @return ClassRef
     * @throws ReflectionException
     */
    public function classRef (object $object): ClassRef {
        $propertiesKey = get_class($object);

        return $this->classRefMap[$propertiesKey] ?? ($this->classRefMap[$propertiesKey] = new ClassRef($object));
    }
}