<?php
/**
 * Created by PhpStorm.
 * User: jiuli
 * Date: 2018/4/9
 * Time: 下午4:22
 */

namespace Mainto\RpcServer\RpcClient;

use Mainto\RpcServer\Exceptions\RpcStreamException;
use Mainto\RpcServer\RpcUtil\RpcBlock;
use Mainto\RpcServer\RpcUtil\RpcProtocol;
use Mainto\RpcServer\RpcUtil\Stream\RpcStreamAbstract;
use Mainto\RpcServer\RpcUtil\Stream\RpcStreamControlBlock;
use Mainto\RpcServer\RpcUtil\Stream\RpcStreamProviderInterface;
use RuntimeException;
use Socket\Raw\Socket;

/**
 * Rpc的流传输是建立在TCP之上的抽象层，该层负责双向的传输
 *
 * Class RpcStream
 * @package Mainto\RpcServer\RpcUtil\Stream
 */
class RpcStream extends RpcStreamAbstract {
    const StreamRawPayloadType = 1;

    /**
     * 流ID
     * @var string
     */
    protected $streamId;

    /**
     * Socket
     * @var Socket
     */
    protected $connection;

    /**
     * 兼容 Workerman 的 Rpc 包装类
     * @var RpcClientSocket
     */
    protected $connectionWarp;

    /**
     * 是否进入流模式
     * @var bool
     */
    protected $isStreamMode = false;

    /**
     * 当前流是否已关闭
     * @var bool
     */
    protected $isClose = false;

    /**
     * stream 内容提供者
     * @var RpcStreamProviderInterface null
     */
    protected $streamProvider = null;

    /**
     * RpcStream constructor.
     * @param Socket $socket
     */
    public function __construct (Socket $socket) {
        $this->streamId = str_random();
        $this->connection = $socket;
        $this->connectionWarp = new RpcClientSocket($this->connection);
    }

    /**
     * 发送第一个数据块
     *
     * @param RpcBlock $block
     * @return RpcStreamResponse|RpcBlock|null
     */
    public function sendFirstBlock (RpcBlock $block) {
        $this->sendToConnection(RpcStreamControlBlock::getInstanceByValue("start", $this->streamId, $block->toArray(), $block->getPayloadType()));

        return $this->processBlock();
    }

    /**
     * 发送控制信息到对端
     *
     * @param RpcStreamControlBlock $block
     */
    private function sendToConnection (RpcStreamControlBlock $block) {
        if (!$this->isClose) {
            $this->connection->write(RpcProtocol::encode($block, $this->connectionWarp));
        } else {
            throw new RpcStreamException("stream is closed");
        }
    }

    /**
     * 处理一个数据块
     *
     * @return RpcStreamResponse|RpcBlock|null
     */
    private function processBlock () {
        $pack = $this->getNextPack();
        if ($pack) {
            switch ($pack->getAction()) {
                case "ready":
                    $this->isStreamMode = true;
                    $response = new RpcStreamResponse($this);
                    $this->streamProvider = $response->getProviderProxy();
                    return $response;
                case "stream":
                    $this->streamProvider->onMessageRecv($this, $pack->getPayload());
                    break;
                case "close":
                    if ($this->isStreamMode) {
                        $this->isClose = true;
                        $this->close();
                        $this->streamProvider->onStreamClosed($pack->getPayload());
                    } else {
                        return RpcBlock::make($pack->getPayloadType(), $pack->getPayload());
                    }
                    break;
            }
        }

        return null;
    }

    /**
     * 从 stream 上获取下一个数据块
     *
     * @return RpcStreamControlBlock|RpcBlock|null
     */
    private function getNextPack () {
        $hasLength = false;
        $len = 4;
        $buf = "";
        while (true) {
            $tmp = $this->connection->read($len - strlen($buf));
            if (strlen($tmp) == 0) {
                throw new RuntimeException("socket read error");
            }
            $buf .= $tmp;

            if (!$hasLength) {
                $len = RpcProtocol::input($buf, $this->connectionWarp);
                if ($len === false) {
                    $this->connection->close();
                    throw new RuntimeException("rpc protocol report data error");
                } else if ($len === 0) {
                    continue;
                } else {
                    $hasLength = true;
                }
            } else {
                if (strlen($buf) >= $len) {
                    $block = RpcProtocol::decode($buf, $this->connectionWarp);
                    return $block;
                }
            }
        }
        return null;
    }

    /**
     * 关闭当前stream
     *
     * @param null $closeMsg
     */
    public function close ($closeMsg = null) {
        if (!$this->isClose) {
            $sendBlock = $this->processResult($closeMsg);
            $this->sendToConnection(RpcStreamControlBlock::getInstance([
                "a" => "close",
                "i" => $this->streamId,
                "p" => $sendBlock->toArray(),
                "t" => $sendBlock->getPayloadType(),
            ]));
            $this->isClose = true;
        }
    }

    /**
     * 当 stream 内容提供者准备就绪时调用
     */
    public function ready () {
        $this->sendToConnection(RpcStreamControlBlock::getInstanceByValue("ready", $this->streamId, null, self::StreamRawPayloadType));
        $this->streamProvider->onStreamStart($this);

        while (!$this->isClose) {
            $this->processBlock();
        }

    }

    /**
     * 发送 stream 数据到对端
     *
     * @param $data
     */
    public function send ($data) {
        $this->sendToConnection(RpcStreamControlBlock::getInstanceByValue("stream", $this->streamId, $this->processResult($data, false), self::StreamRawPayloadType));
    }
}