<?php
/**
 * Created by PhpStorm.
 * User: jiuli
 * Date: 2018/4/4
 * Time: 下午1:29
 */

namespace Mainto\RpcServer\RpcClient;


use Exception;
use Illuminate\Support\Str;
use Mainto\RpcServer\Exceptions\RpcInvokeException;
use Mainto\RpcServer\RpcUtil\Block\InvokeControlBlock;
use Mainto\RpcServer\RpcUtil\Block\SuccessReturnBlock;
use Mainto\RpcServer\RpcUtil\RpcBlock;
use RuntimeException;
use Throwable;

class RpcClass {
    protected static $instances = [];

    /**
     * Rpc 客户端
     * @var RpcClient
     */
    protected $client;
    /**
     * 调用服务名
     * @var string
     */
    protected $serviceName;
    /**
     * 调用类名
     * @var string
     */
    protected $className;
    /**
     * 缓存Key
     * @var string
     */
    protected $cacheKey = "";

    /**
     * RpcClass constructor.
     * @param string $cacheKey
     * @param RpcClient $client
     * @param string $serviceName
     * @param string $className
     */
    protected function __construct (string $cacheKey, ?RpcClient $client, string $serviceName, string $className) {
        $this->cacheKey = $cacheKey;
        $this->client = $client;
        $this->serviceName = $serviceName;
        $this->className = $className;
    }

    /**
     * 获得一个远程调用实例（如果有设置 Mock，获得的是 Mock 实例）
     *
     * @param $serviceName
     * @param $className
     * @return RpcClass
     */
    public static function getClass ($serviceName, $className): RpcClass {
        $key = "{$serviceName}->{$className}";
        if (!isset(self::$instances[$key])) {
            self::$instances[$key] = new RpcClass($key, self::getRemoteClient($serviceName), $serviceName, $className);
        }

        return self::$instances[$key];
    }

    /**
     * 获取远程调用客户端
     *
     * @param $serviceName
     * @return RpcClient
     */
    protected static function getRemoteClient ($serviceName): RpcClient {
        $client = RpcClient::getInstanceByServiceName($serviceName);
        if (!$client) {
            throw new RuntimeException("can not find or connect to server {$serviceName}");
        }

        return $client;
    }

    /**
     * 注册一个 Mock 的类
     *
     * @param $serviceName
     * @param $className
     * @return RpcMockClass
     */
    public static function registerMockClass ($serviceName, $className): RpcMockClass {
        $key = "{$serviceName}->{$className}";
        if (!isset(self::$instances[$key]) || !(self::$instances[$key] instanceof RpcMockClass)) {
            self::$instances[$key] = new RpcMockClass($key, $serviceName, $className);
        }

        return self::$instances[$key];
    }

    /**
     * 反注册一个 Mock 的类
     *
     * @param $serviceName
     * @param $className
     * @return bool
     */
    public static function unregisterMockClass ($serviceName, $className): bool {
        $key = "{$serviceName}->{$className}";
        if (isset(self::$instances[$key]) && self::$instances[$key] instanceof RpcMockClass) {
            unset(self::$instances[$key]);
            return true;
        }

        return false;
    }

    /**
     * 反注册所有的 Mock 类
     *
     * @return bool
     */
    public static function unregisterAllMockClass (): bool {
        foreach (self::$instances as $key => $insClass) {
            if ($insClass instanceof RpcMockClass) {
                unset(self::$instances[$key]);
            }
        }

        return true;
    }

    /**
     * 函数调用
     * @param $name
     * @param $arguments
     * @return RpcStreamResponse|RpcBlock|null
     * @throws Throwable
     */
    public function __call ($name, $arguments) {
        if (count($arguments) == 1) {
            $arguments = $arguments[0] ?? $arguments;
        }
        $invoke = new InvokeControlBlock($this->className, $name, $arguments);
        if ($this->client->isClose()) {
            $this->client->close();
            for ($i = 0; $i < 3; $i++) {
                try {
                    $client = RpcClient::getInstanceByServiceName($this->serviceName);
                    if (!$client) {
                        throw new RuntimeException("can not find or connect to server {$this->serviceName}");
                    }
                    $this->client = $client;
                    break;
                } catch (\Exception $exception) {
                    if ($i == 2) {
                        throw $exception;
                    } else {
                        sleep(3);
                    }
                }
            }
        }

        $block = null;
        try {
            $block = $this->client->sendBlock($invoke);
        } catch (Throwable $e) {
            if (Str::contains($e->getMessage(), "Socket operation failed")) {
                $this->client->close();
            }

            throw $e;
        }

        if ($block instanceof SuccessReturnBlock) {
            return $block->getMessage();
        } elseif ($block instanceof RpcStreamResponse) {
            return $block;
        } else {
            throw new RpcInvokeException($block);
        }
    }

    /**
     * 捕捉异常
     *
     * @param Exception $e
     * @return bool
     */
    private function handleSocketException (Exception $e): bool {
        if ($e->getMessage() == "socket read error") {
            $this->client->close();
            $this->client = RpcClient::getInstanceByServiceName($this->serviceName);
            if ($this->client) {
                return true;
            }
        }

        if ($this->client) $this->client->close();
        if (isset(self::$instances[$this->cacheKey])) {
            unset(self::$instances[$this->cacheKey]);
        }
        return false;
    }
}