<?php
/**
 * Created by PhpStorm.
 * User: jiuli
 * Date: 2018/4/29
 * Time: 下午5:00
 */

namespace Mainto\RpcServer\RpcUtil\Tool;

use Mainto\Bridge\Invokes\CoreGear\Lock;
use Mainto\RpcServer\Exceptions\RpcLockException;
use Mainto\RpcServer\Exceptions\RpcRuntimeException;
use Throwable;

class RpcMutex {
    private $lockName;
    private $lockPassword = null;

    /**
     * RpcLock constructor.
     * @param $lockName
     */
    private function __construct ($lockName) { $this->lockName = $lockName; }

    /**
     * 获取一个锁
     * @param $lockName
     * @return RpcMutex
     */
    public static function getLock ($lockName) {
        return new RpcMutex($lockName);
    }

    /**
     * 对接下来的调用进行锁定
     * @param callable $callback
     * @param int $expireTime
     * @return mixed
     * @throws Throwable
     */
    public function synchronized (callable $callback, int $expireTime = 86400) {
        if ($this->tryLock($expireTime)) {
            try {
                return $callback();
            } catch (Throwable $e) {
                throw $e;
            } finally {
                $this->unlock();
            }
        } else {
            throw new RpcLockException("server busy, try it later", 0xF0106801);
        }
    }

    /**
     * 尝试加锁
     * @param int $expireTime 自动解锁时间
     * @return bool
     * @throws \Exception
     */
    public function tryLock (int $expireTime = 86400): bool {
        $lockPassword = Lock::getLock($this->lockName, $expireTime);
        if ($lockPassword === false) {
            return false;
        }

        $this->lockPassword = $lockPassword;
        return true;
    }

    /**
     * 对当前操作进行解锁
     * @throws \Exception
     */
    public function unlock () {
        if ($this->lockPassword === null) {
            throw new RpcRuntimeException("try to unlock of unlocked RpcMutex");
        }

        if (Lock::unlock($this->lockName, $this->lockPassword)) {
            $this->lockPassword = null;
        } else {
            throw new RpcRuntimeException("try to unlock of expired RpcMutex");
        }
    }
}