<?php
/**
 * Created by PhpStorm.
 * User: jiuli
 * Date: 2018/3/19
 * Time: 下午3:49
 */

namespace Mainto\RpcServer\RpcUtil;

use InvalidArgumentException;
use Mainto\RpcServer\RpcUtil\Stream\RpcStreamControlBlock;
use Throwable;
use Workerman\Connection\ConnectionInterface;
use Workerman\Protocols\ProtocolInterface;

class RpcProtocol implements ProtocolInterface {
    /**
     * 用于在接收到的recv_buffer中分包
     *
     * 如果可以在$recv_buffer中得到请求包的长度则返回整个包的长度
     * 否则返回0，表示需要更多的数据才能得到当前请求包的长度
     * 如果返回false或者负数，则代表错误的请求，则连接会断开
     *
     * @param ConnectionInterface $connection
     * @param string $recv_buffer
     * @return int|false
     */
    public static function input ($recv_buffer, ConnectionInterface $connection) {
        if (strlen($recv_buffer) < 4) { //不足4个则继续接受
            return 0;
        } else {
            $size = self::bigEndianStrToInt(substr($recv_buffer, 0, 4));
            if ($size > 10485760 || $size <= 0) { // 每段不应该大于10M或小于0byte
                return false;
            } else {
                return $size + 4;
            }
        }
    }

    /**
     * 字符串转为二进制整形（大端转换）
     * @param string $str
     * @return int
     */
    private static function bigEndianStrToInt (string $str): int {
        if (strlen($str) < 4) {
            throw new InvalidArgumentException("bigEndianStrToInt: str len is too small");
        }

        return ord($str[3]) | (ord($str[2]) << 8) | (ord($str[1]) << 16) | (ord($str[0]) << 24);
    }

    /**
     * Decode package and emit onMessage($message) callback, $message is the result that decode returned.
     *
     * @param ConnectionInterface $connection
     * @param string $recv_buffer
     * @return RpcStreamControlBlock
     */
    public static function decode ($recv_buffer, ConnectionInterface $connection) {
        $data = msgpack_unpack(substr($recv_buffer, 4));

        try {
            return RpcStreamControlBlock::getInstance($data);
        } catch (Throwable $e) {
            $connection->close();
            return null;
        }
    }

    /**
     * Encode package before sending to client.
     *
     * @param ConnectionInterface $connection
     * @param RpcStreamControlBlock $data
     * @return string
     */
    public static function encode ($data, ConnectionInterface $connection) {
        $data = $data->getData();
        $data = msgpack_pack($data);

        $sizeData = self::bigEndianIntToStr(strlen($data));
        return $sizeData.$data;
    }

    /**
     * 二进制整形转为字符串（大端转换）
     * @param int $int
     * @return string
     */
    private static function bigEndianIntToStr (int $int): string {
        return chr($int >> 24 & 0xFF).chr($int >> 16 & 0xFF).chr($int >> 8 & 0xFF).chr($int & 0xFF);
    }
}