<?php
/**
 * Created by PhpStorm.
 * User: liuzaisen
 * Date: 2019/4/4
 * Time: 11:54 AM
 */

namespace Mainto\RpcServer\TestHelper\Document;


use Carbon\Carbon;
use Doctrine\Common\Annotations\AnnotationReader;
use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use Mainto\RpcServer\Exceptions\RpcRuntimeException;
use Mainto\RpcServer\RpcAnnotations\Alias;
use Mainto\RpcServer\RpcAnnotations\RpcApi;
use Mainto\RpcServer\RpcAnnotations\RpcCallWay;
use Mainto\RpcServer\RpcAnnotations\RpcHeader;
use Mainto\RpcServer\RpcAnnotations\RpcParam;
use ReflectionMethod;

/**
 * Class GroupApi
 * @package Mainto\RpcServer\TestHelper\Document
 * @property string $name
 * @property string $method
 * @property string $comment
 * @property string $edit_time
 * @property string $api_path
 * @property string $bridge_namespace
 * @property string $js_sdk_file_name
 * @property string $controller_name
 * @property string $method_name
 * @property array $params
 * @property Collection $request_example
 */
class GroupApi implements Arrayable, \ArrayAccess {
    /**
     * @var string
     */
    protected $name;

    /**
     * @return string
     */
    public function getName (): string {
        return $this->name;
    }

    /**
     * @return string
     */
    public function getMethod (): string {
        return $this->method;
    }

    /**
     * @return string
     */
    public function getComment (): string {
        return $this->comment;
    }

    /**
     * @return string
     */
    public function getApiPath (): string {
        return $this->api_path;
    }

    /**
     * @return string
     */
    public function getBridgeNamespace (): string {
        return $this->bridge_namespace;
    }

    /**
     * @return string
     */
    public function getJsSdkFileName (): string {
        return $this->js_sdk_file_name;
    }

    /**
     * @return string
     */
    public function getControllerName (): string {
        return $this->controller_name;
    }

    /**
     * @return string
     */
    public function getMethodName (): string {
        return $this->method_name;
    }

    /**
     * @return array
     */
    public function getParams (): array {
        return $this->params;
    }

    /**
     * @return Collection
     */
    public function getRequestExample (): Collection {
        return $this->request_example;
    }

    /**
     * @var string
     */
    protected $method;

    /**
     * @var string
     */
    protected $edit_time;

    /**
     * @var string
     */
    protected $comment;

    /**
     * @var string
     */
    protected $api_path;

    /**
     * @var string
     */
    protected $bridge_namespace;

    /**
     * @var string
     */
    protected $js_sdk_file_name;

    /**
     * @var string
     */
    protected $controller_name;

    /**
     * @var string
     */
    protected $method_name;

    /**
     * @var array
     */
    protected $params;

    /**
     * @var array
     */
    protected $request_example;

    /**
     * Document constructor.
     * @param array $params
     */
    public function __construct ($params = []) {
        foreach ($params as $key => $value) {
            if (property_exists($this, $key)) {
                $this->{$key} = $value;
            }
        }
    }

    /**
     * @param $controllerShortName
     * @param $method
     * @return ReflectionMethod
     */
    protected static function getRefMethod ($controllerShortName, $method) {
        $refMethod = null;
        \RpcRouter::dump(function () use ($controllerShortName, $method, &$refMethod) {
            /** @var ReflectionMethod $refMethod */
            if (isset($this->methodsMap[$controllerShortName.'::'.$method])) {
                $refMethod = $this->methodsMap[$controllerShortName.'::'.$method];
            } else {
                throw new RpcRuntimeException('The called method not found! ');
            }
        });
        return $refMethod;
    }

    public function addRequestExample ($name, $request, $response, $comment = '') {
        /** @var GroupApiRequestExample $example */
        $example = $this->request_example->where('name', $name)->first();
        if (!$example) {
            $this->request_example->push(GroupApiRequestExample::newExample($name, $request, $response, $comment));
        } else {
            $example->setRequest($request);
            $example->setResponse($response);
            $example->setComment($comment);
        }
    }

    /**
     * @throws \Doctrine\Common\Annotations\AnnotationException
     */
    public function flush ($controller_name) {
        $params = self::getBaseInfo($controller_name, $this->method_name);
        foreach ($params as $key => $value) {
            if (property_exists($this, $key)) {
                $this->{$key} = $value;
            }
        }
    }

    /**
     * @param $controllerShortName
     * @param $method
     * @return array
     * @throws \Doctrine\Common\Annotations\AnnotationException
     */
    public static function getBaseInfo ($controllerShortName, $method) {
        $refMethod = self::getRefMethod($controllerShortName, $method);
        $reader = new AnnotationReader();
        $annotations = $reader->getMethodAnnotations($refMethod);
        /** @var RpcApi $rpcApi */
        $rpcApi = null;
        $name = $refMethod->getName();
        $params = []; //[RpcParams]
        $headers = []; //[RpcHeader]
        $bridgeDoc = true;
        $jsSdk = true;
        //get annotation
        foreach ($annotations as $annotation) {
            if ($annotation instanceof RpcCallWay && $annotation->only == 'Http') {
                $bridgeDoc = false;
            }
            if ($annotation instanceof RpcCallWay && $annotation->only == 'Rpc') {
                $jsSdk = false;
            }
            if ($annotation instanceof RpcApi) {
                $rpcApi = $annotation;
            }
            if ($annotation instanceof RpcParam) {
                $params[] = $annotation;
            }
            if ($annotation instanceof Alias) {
                $name = $annotation->name;
            }
            if ($annotation instanceof RpcHeader) {
                $headers[] = $annotation;
            }
        }
        //get doc
        $comment = '';
        $docComment = $refMethod->getDocComment();
        foreach (explode("\n", $docComment) as $line) {
            if (strpos($line, '@') === false && strpos($line, '/') === false) {
                $_doc = trim(str_replace('*', '', $line));
                if (!$_doc) continue;
                $comment .= $_doc."  \n";
                continue;
            }
        }
        $class_name = $refMethod->getDeclaringClass()->getName();
        $micro_name = config('rpc-server.service_name');
        $methodStr = trim($micro_name.'\\'.substr(self::getNeedClassPath($class_name), 0, -10));
        $uc_micro_name = Str::studly($micro_name);
        $controllerName = str_replace('\\', '_', substr(self::getNeedClassPath($class_name), 0, -10));

        $endName = $controllerShortName;
        if (($pos = strrpos($controllerShortName, "\\")) !== false) {
            $endName = substr($controllerShortName, $pos + 1);
        }

        return [
            'name'             => $name,
            'method'           => $rpcApi ? $rpcApi->method : null,
            'comment'          => $comment,
            'edit_time'        => Carbon::now()->toDateTimeString(),
            'api_path'         => $rpcApi ? ('/'.Str::snake($micro_name).$rpcApi->url) : null,
            'bridge_namespace' => $bridgeDoc ? ("\Mainto\Bridge\Invokes\\$methodStr") : null,
            'js_sdk_file_name' => $rpcApi && $jsSdk ? ("mainto-jssdk/interface/micro_{$uc_micro_name}_$controllerName.js") : null,
            'controller_name'  => str_replace('Controller', '', $endName),
            'method_name'      => $method,
            'params'           => self::parseParams($params),
        ];
    }

    /**
     * @param $controllerShortName
     * @param $method
     * @return GroupApi
     * @throws \Doctrine\Common\Annotations\AnnotationException
     */
    public static function newGroupApi ($controllerShortName, $method) {
        return new self(
            array_merge(
                self::getBaseInfo($controllerShortName, $method),
                [
                    'request_example' => collect([]),
                ]
            )
        );
    }

    /**
     * @param $params
     * @return array
     */
    protected static function parseParams ($params) {
        $list = [];
        /** @var RpcParam $param */
        foreach ($params as $param) {
            $required = false;
            if ($param->require == true || in_array('required', explode('|', $param->validation))) {
                $required = true;
            }
            $param->validation =
            $list[] = [
                'name'         => $param->name,
                'type'         => $param->type,
                'must_include' => $required,
                'comment'      => $param->comment,
                'other_info'   => str_replace('|', '/', $param->validation),
                'example'      => $param->example,
            ];
        }
        return $list;
    }

    public function toArray () {
        return [
            'name'             => $this->name,
            'method'           => $this->method,
            'comment'          => $this->comment,
            'edit_time'        => $this->edit_time,
            'api_path'         => $this->api_path,
            'bridge_namespace' => $this->bridge_namespace,
            'js_sdk_file_name' => $this->js_sdk_file_name,
            'controller_name'  => $this->controller_name,
            'method_name'      => $this->method_name,
            'params'           => $this->params,
            'request_example'  => $this->request_example,
        ];
    }

    private static function getNeedClassPath ($className) {
        return str_replace('\Rpc\Controllers\\', '', strstr($className, '\Rpc\Controllers\\'));
    }

    /**
     * Whether a offset exists
     * @link  http://php.net/manual/en/arrayaccess.offsetexists.php
     * @param mixed $offset <p>
     *                      An offset to check for.
     *                      </p>
     * @return boolean true on success or false on failure.
     *                      </p>
     *                      <p>
     *                      The return value will be casted to boolean if non-boolean was returned.
     * @since 5.0.0
     */
    public function offsetExists ($offset) {
        return property_exists($this, $offset);
    }

    /**
     * Offset to retrieve
     * @link  http://php.net/manual/en/arrayaccess.offsetget.php
     * @param mixed $offset <p>
     *                      The offset to retrieve.
     *                      </p>
     * @return mixed Can return all value types.
     * @since 5.0.0
     */
    public function offsetGet ($offset) {
        if (property_exists($this, $offset)) {
            return $this->{$offset};
        }
    }

    /**
     * Offset to set
     * @link  http://php.net/manual/en/arrayaccess.offsetset.php
     * @param mixed $offset <p>
     *                      The offset to assign the value to.
     *                      </p>
     * @param mixed $value  <p>
     *                      The value to set.
     *                      </p>
     * @return void
     * @since 5.0.0
     */
    public function offsetSet ($offset, $value) {
        // TODO: Implement offsetSet() method.
    }

    /**
     * Offset to unset
     * @link  http://php.net/manual/en/arrayaccess.offsetunset.php
     * @param mixed $offset <p>
     *                      The offset to unset.
     *                      </p>
     * @return void
     * @since 5.0.0
     */
    public function offsetUnset ($offset) {
        // TODO: Implement offsetUnset() method.
    }
}