<?php


namespace Mainto\RpcServer\RpcUtil\Setting;


use Illuminate\Contracts\Support\Arrayable;
use Mainto\RpcServer\RpcUtil\Tool\AgentToolHelper;
use ReflectionClass;
use RuntimeException;

abstract class RpcSetting implements Arrayable {
    /**
     * RpcSetting constructor.
     */
    protected function __construct () {
        if (!($this instanceof RpcSettingProxy))
            throw new RuntimeException("can not construct this instance");
    }

    /**
     * 判断是否强制使用默认配置
     *
     * @return bool
     */
    protected static function isUseDefault (): bool {
        static $cache = null;

        if ($cache === null) {
            $cache = (bool)(
                config('app.env') == "testing" ||
                !config('rpc-server.agent.enable_setting_client') ||
                env("FORCE_SETTING_DEFAULT")
            );
        }

        return $cache;
    }

    /**
     * 新建配置实例
     *
     * @param bool $forceRefresh
     * @return static
     * @throws \ReflectionException
     */
    public static function newInstance ($forceRefresh = false) {
        static $defaultCache = [];
        static $proxyCache = [];
        $settingName = class_basename(static::class);

        if (!isset($defaultCache[$settingName])) {
            $refClass = new ReflectionClass(static::class);
            $props = $refClass->getProperties();
            $defaultInstance = $refClass->newInstanceWithoutConstructor();
            $defaultSetting = [];

            foreach ($props as $prop) {
                $defaultSetting[$prop->getName()] = $prop->getValue($defaultInstance);
            }

            $defaultCache[$settingName] = $defaultSetting;
        }

        if (self::isUseDefault()) {
            return new RpcSettingProxy(1, $defaultCache[$settingName]);
        }

        if (isset($proxyCache[$settingName]) && !$forceRefresh) {
            /** @var RpcSettingProxy $proxy */
            $proxy = $proxyCache[$settingName];
            if (AgentToolHelper::checkConfigVersion($settingName, $proxy->getVersion())) {
                return $proxy;
            }
        }

        $config = AgentToolHelper::getConfig($settingName);
        $proxy = new RpcSettingProxy($config['version'], array_merge($defaultCache[$settingName], $config['content']));
        $proxyCache[$settingName] = $proxy;
        return $proxy;
    }

    /**
     * get config version
     *
     * @return int
     */
    public function getVersion (): int {
        return -1;
    }

    /**
     * Get the instance as an array.
     *
     * @return array
     */
    public function toArray () {
        return [];
    }
}