<?php
/**
 * Created by PhpStorm.
 * User: jiuli
 * Date: 2018/3/16
 * Time: 上午10:59
 */

namespace Mainto\RpcServer\RpcServer;


use Generator;
use Mainto\RpcServer\Base\Controller;
use Mainto\RpcServer\RpcUtil\RpcBlock;
use Mainto\RpcServer\RpcUtil\RpcGeneratorStreamProxy;
use Mainto\RpcServer\RpcUtil\RpcParameter;
use Mainto\RpcServer\RpcUtil\Stream\RpcStreamProviderInterface;
use ReflectionMethod;
use RuntimeException;
use Throwable;
use Workerman\Connection\TcpConnection;

class RpcInvoke {
    /**
     * 当前的 Invoke 链接
     * @var RpcStream
     */
    private $stream;

    /**
     * 当前的 Invoke 类
     * @var Controller
     */
    private $obj;

    /**
     * 当前的 Invoke 方法
     * @var ReflectionMethod
     */
    private $method;

    /**
     * 当前的 Invoke 方法的参数
     * @var array
     */
    private $params;

    /**
     * 当前参数的映射列表
     * @var RpcParameter
     */
    private $paramMap;

    /**
     * 当前的请求ID
     * @var string
     */
    private $curRequestId;

    /**
     * 要发送给远程的信息
     * @var mixed
     */
    private $data;

    /**
     * 调用的 RpcBlock
     * @var RpcBlock
     */
    private $invokeBlock;

    /**
     * 是否是 raw mode
     * @var bool
     */
    private $rawMode;

    /**
     * RpcInvoke constructor.
     * @param RpcStream $stream
     * @param Controller $obj
     * @param ReflectionMethod $method
     * @param array $params
     * @param RpcParameter $paramMap
     * @param RpcBlock $invokeBlock
     * @param bool $rawMode
     */
    public function __construct ($stream, Controller $obj, ReflectionMethod $method, array $params, $paramMap = null, $invokeBlock = null, $rawMode = false) {
        $this->stream = $stream;
        $this->obj = $obj;
        $this->method = $method;
        $this->params = $params;
        $this->curRequestId = str_random();
        $this->paramMap = $paramMap;
        $this->invokeBlock = $invokeBlock;
        $this->rawMode = $rawMode;

        $this->checkConnectionStatus();
    }

    private function checkConnectionStatus () {

    }

    /**
     * 调用并返回调用结果
     *
     * @param RpcContext|null $request
     * @return mixed
     * @throws Throwable
     */
    public function invoke (?RpcContext &$request = null) {
        $retData = null;
        try {
            $request = new RpcContext($this, $this->curRequestId, $this->params, $this->invokeBlock, $this->rawMode);
            $ret = null;

            if ($this->paramMap) {
                $ret = $this->method->invokeArgs($this->obj, $this->paramMap->buildInvokeParam($request));
            } else {
                $ret = $this->method->invoke($this->obj);
            }

            $request->sendReturnToRemote($ret);
            $request->saveSession();
        } finally {
            $this->curRequestId = str_random();
            $retData = $this->data;
        }

        return $retData;
    }

    /**
     * 设置返回的内容
     * @param string $id Request ID
     * @param $data
     */
    public function sendReturnToRemote ($id, $data) {
        if ($id !== $this->curRequestId) {
            throw new RuntimeException("send data must be in sync");
        }

        if ($data instanceof Generator) {
            $this->startStreamMode(new RpcGeneratorStreamProxy($data));
        } else {
            $this->data = $data;
        }
    }

    /**
     * 启动流模式
     * @param RpcStreamProviderInterface $provider
     */
    public function startStreamMode (RpcStreamProviderInterface $provider) {
        $this->stream->setStreamMode($provider);
    }

    /**
     * 获取所在的类名
     *
     * @return string
     */
    public function getClassName () {
        return $this->method->getDeclaringClass()->getShortName();
    }

    /**
     * 获取所在的方法名
     *
     * @return string
     */
    public function getMethodName () {
        return $this->method->getShortName();
    }

    /**
     * 当信息已发送
     * @param TcpConnection $connection
     */
    public function onMessageSend (TcpConnection $connection) {
        if ($this->data instanceof RpcStream) {
            $this->data->readySteam($connection);
        }
    }

    /**
     * @return RpcParameter
     */
    public function getParamMap (): RpcParameter {
        return $this->paramMap;
    }

    /**
     * 获取当前调用方法的名称
     * @return string
     */
    public function getMethodSource (): string {
        return "{$this->method->class}::{$this->method->name}";
    }

    /**
     * 获得当前调用的参数
     */
    public function getParams () {
        return $this->params;
    }
}