<?php

namespace Mainto\RpcServer\Command;

use Doctrine\Common\Annotations\AnnotationReader;
use Illuminate\Console\Command;
use Mainto\RpcServer\RpcAnnotations\Alias;
use Mainto\RpcServer\RpcAnnotations\RpcApi;
use Mainto\RpcServer\RpcAnnotations\RpcCallWay;
use Mainto\RpcServer\RpcAnnotations\RpcParam;
use ReflectionMethod;
use RpcRouter;

class RpcJsSdkDump extends Command {
    public const BASE_DIR_NAME = 'mainto-micro-jssdk';

    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'rpc:dump-mainto-sdk-js {--build-version} {--branch=} {--dump_path=} {--base_namespace=}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'RPC Router Dumper';

    /**
     * Dump IDE Start
     * ==========================================================
     */
    private $methodsMap = [];
    private $classesMap = [];
    /**
     * ===========================================================
     * Dump IDE End
     */
    public $dump_path;
    public $base_namespace;

    public static function versionPlus ($version, $branch) {
        if ($branch && $branch != 'dev') {
            if (strpos($version, 'beta') === false) {
                $branch = str_replace(['-', '_', '/'], '.', $branch);
                $version = $version.'-beta.'.$branch.'.0';
            }
        }
        $versionArr = explode('.', $version);
        $versionArr[count($versionArr) - 1]++;

        return implode('.', $versionArr);
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     * @throws \Doctrine\Common\Annotations\AnnotationException
     * @throws \Throwable
     */
    public function handle () {
        $dump_path = $this->option('dump_path');
        $base_namespace = $this->option('base_namespace');
        $this->dump_path = rtrim($dump_path ?: format_path('.'), '/').'/';
        $this->base_namespace = $base_namespace ?: 'App\Rpc\Controllers\\';
        $build_version = $this->option('build-version');
        if ($build_version) {
            $config = file_get_contents($this->dump_path.(self::BASE_DIR_NAME.'/package.json'));
            $config = json_decode($config, true);
            $config['version'] = self::versionPlus($config['version'], $this->option('branch'));
            file_put_contents($this->dump_path.(self::BASE_DIR_NAME.'/package.json'), json_encode($config, JSON_PRETTY_PRINT));
            dump("new jssdk version {$config['version']}");
            return true;
        }
        $microName = studly_case(config('rpc-server.service_name', 'NoName'));

        if (file_exists($this->dump_path.(self::BASE_DIR_NAME."/source/micro/{$microName}"))) {
            del_file_tree($this->dump_path.(self::BASE_DIR_NAME."/source/micro/{$microName}"));
        }
        mkdir($this->dump_path.(self::BASE_DIR_NAME."/source/micro/{$microName}"), 0777, true);
        $reader = new AnnotationReader();
        $that = &$this;
        RpcRouter::dump(function () use (&$that, $microName, $reader) {
            $dumpMap = [];
            foreach ($this->methodsMap as $name => $value) {
                /** @var ReflectionMethod $method */
                $method = $this->methodsMap[$name];
                [$className, $methodName] = explode("::", $name, 2);
                $shortClassName = $className;

                // get short class name
                if (($pos = strrpos($className, '\\')) !== false) {
                    $shortClassName = substr($className, $pos + 1);
                }

                //gen method blade

                //parse params
                $methodBlade = [];
                $methodBlade['alias'] = $methodName;
                $methodBlade['name'] = $methodName;
                $methodBlade['isDeprecated'] = false;
                $methodBlade['params'] = [];
                $continue = false;
                $hasUrl = false;
                $annotations = $reader->getMethodAnnotations($method);
                foreach ($annotations as $annotation) {
                    if ($annotation instanceof RpcCallWay && $annotation->only == 'Rpc') {
                        $continue = true;
                        break;
                    }

                    if ($annotation instanceof Alias) {
                        $methodBlade['alias'] = $annotation->name;
                    }

                    if ($annotation instanceof RpcApi) {
                        $methodBlade['url'] = $annotation->url;
                        $methodBlade['method'] = $annotation->method;
                        $hasUrl = true;
                    }

                    if ($annotation instanceof RpcParam) {
                        $tempType = $annotation->type;
                        if ($annotation->type == "uint") $annotation->type = "int";
                        if (in_array($annotation->type, ["double", "unsignedDouble"])) $annotation->type = "float";
                        if (in_array($annotation->type, ["array", "array<int>", "array<uint>", "array<string>"])) $annotation->type = "array";

                        if ($annotation->require == true || in_array('required', explode('|', $annotation->validation))) {
                            $annotation->require = true;
                            $str = "{$annotation->type} {$annotation->name}";
                            $paramStrArr[] = $str;
                        }
                        if ($tempType != $annotation->type) {
                            $annotation->comment .= sprintf(" [Original type %s]", $tempType);
                        }

                        $methodBlade['params'][] = $annotation;
                    }
                }
                if ($continue) continue;
                if (!$hasUrl) continue;

                //find deprecated and return
                $returnType = '';
                $docComment = $method->getDocComment();
                foreach (explode("\n", $docComment) as $line) {
                    if (!$methodBlade['isDeprecated'] && strpos($line, '@deprecated') !== false) {
                        $methodBlade['isDeprecated'] = true;
                    }
                    if (!$returnType && strpos($line, '@return') !== false) {
                        $returnType = trim(str_replace(['@return', '*'], '', $line));
                    }
                }
                if (!in_array($returnType, ['int', 'array', 'string', 'float'])) {
                    $returnType = 'mixed';
                }
                $methodBlade['returnType'] = $returnType;
                $namespace = explode('\\', str_replace($that->base_namespace, '', get_class($this->classesMap[$className])));
                array_pop($namespace);
                $dumpMap[substr($className, 0, -10)]['methods'][] = $methodBlade;
                $dumpMap[substr($className, 0, -10)]['namespace'] = implode('/', $namespace);
                $dumpMap[substr($className, 0, -10)]['shortName'] = substr($shortClassName, 0, -10);
            }

            foreach ($dumpMap as $className => $item) {
                $path = $that->dump_path.(RpcJsSdkDump::BASE_DIR_NAME."/source/micro/{$microName}/{$item['namespace']}");
                if (!is_dir($path)) {
                    mkdir($path, 0777, true);
                }
                $that->comment(">>>> write /source/micro/{$microName}/{$item['namespace']}/{$item['shortName']}.js");

                $item['namespace'] = $item['namespace'] ? $microName.'\\'.$item['namespace'] : $microName;
                $filename = "$path/{$item['shortName']}.js";

                file_put_contents("$filename", view('js-sdk', [
                    'microName'      => $microName,
                    'className'      => $className,
                    'shortClassName' => $item['shortName'],
                    'methods'        => $item['methods'],
                    'namespace'      => $that->base_namespace.str_replace('/', '\\', $item['namespace']),
                ])->render());

                // 删除最后一个方法的逗号
                $content = file_get_contents($filename);
                $last = strrpos($content, ',');
                $new_content = substr($content, 0, $last);
                $new_content .= substr($content, $last + 1, strlen($content));
                file_put_contents($filename, $new_content);
            }
        });
        return 0;
    }
}
