<?php
/**
 * Created by PhpStorm.
 * User: PHPStorm
 * Date: 18-12-23
 * Time: 下午4:51
 */

namespace Mainto\RpcServer\RpcServer;

use ErrorException;
use Exception;
use Illuminate\Support\Facades\Log;
use Mainto\RpcServer\RpcUtil\EtcdClient;
use Mainto\RpcServer\RpcUtil\RpcProtocol;
use Mainto\RpcServer\RpcUtil\Stream\RpcStreamControlBlock;
use Throwable;
use Workerman\Connection\TcpConnection;
use Workerman\Lib\Timer;
use Workerman\Worker;

class RpcServerWorkerMode extends RpcServer {
    /**
     * RpcServer 实例
     * @var RpcServer
     */
    private static $instance = null;

    /**
     * 是否在后台运行
     *
     * @var bool
     */
    private $daemon;

    /**
     * 监听地址
     *
     * @var string
     */
    private $listenAddress = "";

    /**
     * Worker 数目
     *
     * @var int
     */
    private $workerNum = 1;

    /**
     * Worker
     *
     * @var Worker
     */
    private $worker = null;

    /**
     * Etcd客户端（仅在 0 Worker 中有效）
     *
     * @var EtcdClient
     */
    private $etcdClient = null;

    /**
     * Etcd Lease Id（仅在 0 Worker 中有效）
     *
     * @var null|int
     */
    private $etcdLeaseId = null;
    /**
     * kubernetes mode
     * @var bool
     */
    private $kubernetesMode = false;

    /**
     * RpcServer constructor.
     *
     * @param bool $daemon
     */
    private function __construct ($daemon = false) {
        parent::__construct();

        $address = config('rpc-server.listen');
        $port = config('rpc-server.port');

        $this->daemon = $daemon;
        $this->listenAddress = "tcp://{$address}:{$port}";
        $this->workerNum = config('rpc-server.work_num');
        $this->kubernetesMode = !!env('KUBERNETES_SERVICE_HOST');
        self::$verboseMode = config('rpc-server.verbose_mode');

        Log::info("init workerNum = {$this->workerNum}");
    }

    /**
     * 运行
     *
     * @param string $command
     * @param bool $daemon
     */
    public static function exec (string $command, bool $daemon = true) {
        if (self::$instance == null) {
            self::$instance = new RpcServerWorkerMode($daemon);
        }

        self::$instance->run($command);
    }

    /**
     * 命令行开始执行
     * @param $command
     */
    public function run ($command) {
        $this->initParam($command);
        Log::info("Listen: {$this->listenAddress}");

        Worker::$logFile = storage_path('logs/workman.log');
        $this->worker = new Worker($this->listenAddress);
        $this->worker->count = $this->workerNum;
        //$this->worker->reusePort = true;
        $this->worker->protocol = RpcProtocol::class;

        // Emitted when worker processes start.
        $this->worker->onWorkerStart = function ($worker) {
            if ($worker->id === 0) {
                $this->init();
            }
        };

        $this->worker->onWorkerStop = function ($worker) {
            if ($worker->id === 0) {
                $this->stop();
            }
        };

        // Emitted when new connection come
        $this->worker->onConnect = function (TcpConnection $connection) {
            $this->onConnect($connection);
        };

        // Emitted when data received
        $this->worker->onMessage = function (TcpConnection $connection, $data) {
            if ($data) {
                $this->onMessage($connection, $data);
            }
        };

        // Emitted when new connection come
        $this->worker->onClose = function (TcpConnection $connection) {
            $this->onClose($connection);
        };

        Worker::runAll();
    }

    /**
     * 初始化命令行参数
     * @param $command
     */
    private function initParam ($command) {
        global $argv, $argc;
        $argv = [$argv[0], $command];
        if ($this->daemon) {
            $argv[] = '-d';
        }

        $argc = count($argv);
    }

    /**
     * 初始化（只会被执行一次）
     * @param bool $k8s_mode
     */
    private function init ($k8s_mode = false) {
        if (!$k8s_mode && $this->kubernetesMode) {
            // k8s下启动后3秒后再注册
            Timer::add(3, function () {
                $this->init(true);
            }, [], false);
            return;
        }
        try {
            $serviceName = config('rpc-server.service_name');
            $hostname = $this->getHostName();
            $this->registerService($serviceName, $hostname);

            Timer::add(15, function () use ($serviceName, $hostname) {
                try {
                    $this->etcdClient->keepAlive($this->etcdLeaseId);
                } catch (ErrorException $e) {
                    if (strpos($e->getMessage(), "Undefined index") !== false) {
                        $this->registerService($serviceName, $hostname);
                    }
                } catch (Exception $e) {
                    Log::error($e);

                    Log::info("etcd error, try exit and restart...");

                    // try exit and restart
                    exec("php {$_SERVER['PHP_SELF']} rpc:server stop");
                    Worker::stopAll();
                }
            });

            Log::info("init ".self::SERVICE_PREFIX."{$serviceName}/{$hostname} ok!");
        } catch (Exception $e) {
            Log::error($e);

            sleep(3);

            exec("php {$_SERVER['PHP_SELF']} rpc:server stop");
            Worker::stopAll();
        }
    }

    private function getHostName () {
        if ($this->kubernetesMode) {
            // kubernetes模式
            $hostname = gethostname();
            $hostBlock = explode('-', $hostname);
            $env = array_last(array_slice($hostBlock, 0, count($hostBlock) - 2));
            $chartName = implode('-', array_slice($hostBlock, 0, count($hostBlock) - 3));

            return "{$chartName}-{$env}-{$chartName}";
        } else {
            // 通用模式
            return gethostname();
        }
    }

    private function registerService ($serviceName, $hostname) {
        $config = array_merge(RpcRouter::getBaseRouterConfig(), [
            'addresses'      => $this->getCurrentIp(),
            'port'           => config('rpc-server.port'),
            'max_connection' => $this->workerNum * 2,
        ]);

        $this->etcdClient = get_etcd_client();

        $leaseId = $this->etcdClient->grant(30);
        $this->etcdLeaseId = intval($leaseId['ID']);
        $this->etcdClient->put(self::SERVICE_PREFIX."{$serviceName}/{$hostname}", json_encode($config), [
            'lease' => $this->etcdLeaseId,
        ]);
    }

    private function getCurrentIp () {
        if (config('rpc-server.local_host_address')) {
            // 本地模式
            return [config('rpc-server.local_host_address')];
        } elseif ($this->kubernetesMode) {
            // kubernetes模式
            $hostname = $this->getHostName();

            return array_values(array_unique(array_values(array_merge(
                [$hostname],
                gethostbynamel($hostname)
            ))));
        } else {
            // 通用模式
            $hostname = gethostname();
            return array_values(array_unique(array_values(array_merge(
                [$hostname],
                gethostbynamel($hostname) ?: [],
                try_get_ips()
            ))));
        }
    }

    /**
     * 被停止（只会被执行一次，不保证100%被执行）
     */
    public function stop () {
        try {
            if (!$this->kubernetesMode) {
                // kubernetes 上不需要注销，由服务负责维护及重定向
                $this->etcdClient->revoke($this->etcdLeaseId);
            }

            Log::info("revoke service ok!");
        } catch (Exception $e) {
            Log::error($e);
        }
    }

    /**
     * 当新的链接建立时
     *
     * @param TcpConnection $connection
     */
    private function onConnect (TcpConnection $connection) {
        if (self::$verboseMode) {
            Log::debug("New Connect: {$connection->getRemoteAddress()}");
        }
    }

    /**
     * 当链接消息送达时
     *
     * @param TcpConnection $connection
     * @param RpcStreamControlBlock $block
     *
     * @throws Throwable
     */
    private function onMessage (TcpConnection $connection, RpcStreamControlBlock $block) {
        try {
            $steam = RpcStream::getStreamByStreamId($connection, $block->getStreamId());
            $steam->push($block);
        } catch (Throwable $e) {
            $connection->close();
            Log::error($e);
            throw $e;
        }
    }

    /**
     * 当链接被关闭时
     *
     * @param TcpConnection $connection
     */
    private function onClose (TcpConnection $connection) {
        RpcStream::closeAllStream($connection);
        if (self::$verboseMode) {
            Log::debug("Connect Close: {$connection->getRemoteAddress()}");
        }
    }
}