<?php
/**
 * Created by PhpStorm.
 * User: PHPStorm
 * Date: 18-10-24
 * Time: 上午10:18
 */

namespace Mainto\RpcServer\RpcUtil\Tool\LogDriver;


use Aliyun\SLS\Client;
use Aliyun\SLS\Models\LogItem;
use Aliyun\SLS\Requests\PutLogsRequest;
use Illuminate\Support\Facades\Log;
use RuntimeException;
use Throwable;

class AliyunSLS extends LogDriveAbstract {

    private $logClient;
    private $topic;
    private $project;

    public function __construct () {
        parent::__construct();

        $this->logClient = new Client(
            config("rpc-server.log_config.aliyun.endpoint"),
            config("rpc-server.log_config.aliyun.accessKeyId"),
            config("rpc-server.log_config.aliyun.accessKey")
        );

        $this->topic = config("rpc-server.service_name");
        $this->project = config("rpc-server.log_config.aliyun.project");

        if ($this->localMode) {
            throw new RuntimeException('aliyun sls log driver can not support local mode');
        }
    }

    /**
     * log throwable error log
     *
     * @param $request
     * @param $source
     * @param Throwable $e
     */
    public function logThrowable ($request, Throwable $e, string $source = "") {
        if ($source == "") {
            $traceInfo = $this->getTraceInfo(-1);
            $source = $traceInfo['source'];
        }

        $this->send($source, "error", [
            "request"       => $this->transObject($request),
            "error_name"    => class_basename($e),
            "error_code"    => $e->getCode(),
            "error_content" => $e->getMessage(),
            "error_file"    => $e->getFile(),
            "error_line"    => $e->getLine(),
            "error_trace"   => "\n".$e->getTraceAsString(),
        ]);
    }


    /**
     * send message to aliyun log service
     *
     * @param $source
     * @param $logStore
     * @param $contents
     */
    private function send (string $source, string $logStore, $contents) {
        try {
            $logItem = new LogItem();
            $logItem->setTime(time());
            $logItem->setData($contents);
            $req = new PutLogsRequest($this->project, $logStore, $this->topic, $source, [$logItem]);
            $this->logClient->putLogs($req);
        } catch (Throwable $e) {
            Log::error("rpc_log: ", $contents);
            Log::info($e);
        }
    }

    /**
     * log context
     *
     * @param $source
     * @param array $request
     * @param array $response
     */
    public function logContext ($request, $response, string $source = "") {
        if ($source == "") {
            $traceInfo = $this->getTraceInfo(-1);
            $source = $traceInfo['source'];
        }

        $this->send($source, "context", [
            "request"  => $this->transObject($request),
            "response" => $this->transObject($response),
        ]);
    }

    /**
     * log info message
     *
     * @param $title
     * @param $message
     */
    public function info (string $title, $message) {
        $this->log("INFO", $title, $message);
    }

    /**
     * log some messgae
     *
     * @param string $level
     * @param string $title
     * @param $message
     */
    protected function log (string $level, string $title, $message) {
        $traceInfo = $this->getTraceInfo();
        $this->send($traceInfo['source'], "log", [
            "offset"  => $traceInfo['file'],
            "level"   => $level,
            "title"   => $title,
            "content" => $this->transObject($message),
        ]);
    }

    /**
     * log debug message
     *
     * @param $title
     * @param $message
     */
    public function debug (string $title, $message) {
        $this->log("DEBU", $title, $message);
    }

    /**
     * log notice message
     *
     * @param $title
     * @param $message
     */
    public function notice (string $title, $message) {
        $this->log("NOTI", $title, $message);
    }

    /**
     * log warning message
     *
     * @param $title
     * @param $message
     */
    public function warning (string $title, $message) {
        $this->log("WARN", $title, $message);
    }

    /**
     * log error message
     *
     * @param $source
     * @param $title
     * @param $message
     */
    public function error (string $title, $message) {
        $this->log("ERRO", $title, $message);
    }

    /**
     * log critical message
     *
     * @param $title
     * @param $message
     */
    public function critical (string $title, $message) {
        $this->log("CRIT", $title, $message);
    }
}