<?php

namespace Mainto\RpcServer\Command;

use Doctrine\Common\Annotations\AnnotationReader;
use Illuminate\Console\Command;
use Mainto\RpcServer\RpcAnnotations\Alias;
use Mainto\RpcServer\RpcAnnotations\RpcCallWay;
use Mainto\RpcServer\RpcAnnotations\RpcParam;
use Mainto\RpcServer\RpcServer\RpcContext;
use ReflectionMethod;
use RpcRouter;

class RpcBridgeDump extends Command {
    use RpcEnumDump;

    public $hasError = false;
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'rpc:dump {--dump_path=} {--write_json=} {--base_namespace=} {--base_enum_namespace=}';
    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'RPC Router Dumper bridge';
    /**
     * Dump IDE Start
     * ==========================================================
     */
    private $methodsMap = [];
    private $classesMap = [];
    /**
     * ===========================================================
     * Dump IDE End
     */

    private $microName = "";

    public $dump_path;
    public $base_namespace;

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct () {
        parent::__construct();
        $this->microName = config('rpc-server.service_name', 'NoName');
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     * @throws \Doctrine\Common\Annotations\AnnotationException
     * @throws \ReflectionException
     * @throws \Throwable
     */
    public function handle () {
        $dump_path = $this->option('dump_path');
        $write_json = $this->option('write_json');
        $base_namespace = $this->option('base_namespace');
        $this->dump_path = $dump_path ?: format_path('.');
        $this->base_namespace = $base_namespace ?: 'App\Rpc\Controllers\\';

        $base_enum_namespace = $this->option('base_enum_namespace');
        $this->base_enum_namespace = $base_enum_namespace ?: 'App\Enums';

        $microName = camel_case(config('rpc-server.service_name', 'NoName'));
        $microName[0] = strtoupper($microName[0]);
        $this->dumpEnum('micro-bridge', 'Bridge', $microName);

        if (file_exists($this->dump_path."/micro-bridge/src/Invokes/{$microName}")) {
            del_file_tree($this->dump_path."/micro-bridge/src/Invokes/{$microName}");
        }
        mkdir($this->dump_path."/micro-bridge/src/Invokes/{$microName}", 0777, true);
        $reader = new AnnotationReader();
        $that = &$this;
        RpcRouter::dump(function () use (&$that, $microName, $reader) {
            $dumpMap = [];
            foreach ($this->methodsMap as $name => $value) {
                /** @var ReflectionMethod $method */
                $method = $this->methodsMap[$name];
                [$className, $methodName] = explode("::", $name, 2);
                $shortClassName = $className;

                // get short class name
                if (($pos = strrpos($className, '\\')) !== false) {
                    $shortClassName = substr($className, $pos + 1);
                }

                //gen method blade

                //parse params
                $methodBlade = [];
                $methodBlade['doc'] = [];
                $methodBlade['name'] = $methodName;
                $methodBlade['params'] = [];
                $methodBlade['isDeprecated'] = false;
                $alias = "";
                $disable = false;
                $annotations = $reader->getMethodAnnotations($method);
                foreach ($annotations as $annotation) {
                    if ($annotation instanceof RpcCallWay && $annotation->only == 'Http') {
                        $disable = true;
                        break;
                    }

                    if ($annotation instanceof RpcParam) {
                        $tempType = $annotation->type;
                        if ($annotation->type == "uint") $annotation->type = "int";
                        if (in_array($annotation->type, ["double", "unsignedDouble"])) $annotation->type = "float";
                        if (in_array($annotation->type, ["array", "array<int>", "array<uint>", "array<string>"])) $annotation->type = "array";

                        if ($annotation->require == true || in_array('required', explode('|', $annotation->validation))) {
                            $annotation->require = true;
                        }
                        if ($tempType != $annotation->type) {
                            $annotation->comment .= sprintf(" [Original type %s]", $tempType);
                        }

                        $methodBlade['params'][$annotation->name] = $annotation;
                    } elseif ($annotation instanceof Alias) {
                        $alias = $annotation->name;
                    }
                }
                if ($disable) continue;

                //find deprecated and return
                $returnType = '';
                $docComment = $method->getDocComment();
                $args = $method->getParameters();
                $methodBlade['argStr'] = '';
                $methodBlade['args'] = [];
                foreach ($args as $arg) {
                    if ($arg->getType() && $arg->getType()->getName() == RpcContext::class) continue;
                    $_name = $arg->getName();

                    if (!isset($methodBlade['params'][$_name])) {
                        $that->error('The param not defined in RpcParam: '.$methodName.' : '.$_name);
                        $that->hasError = true;
                        continue;
                    }
                    $type = $methodBlade['params'][$_name]->type;
                    $_argStr = $type.' $'.$_name;
                    $_argStr .= $arg->isDefaultValueAvailable() ? ' = '.var_export_min($arg->getDefaultValue()) : '';

                    $methodBlade['args'][] = $_name;
                    $methodBlade['argStr'] .= $_argStr;
                    $methodBlade['argStr'] .= ', ';
                }
                $methodBlade['argStr'] = trim($methodBlade['argStr'], ', ');
                foreach (explode("\n", $docComment) as $line) {
                    if (strpos($line, '@') === false && strpos($line, '/') === false) {
                        $_doc = trim(str_replace('*', '', $line));
                        if (!$_doc) continue;
                        $methodBlade['doc'][] = $_doc;
                        continue;
                    }

                    if (!$methodBlade['isDeprecated'] && strpos($line, '@deprecated') !== false) {
                        $methodBlade['isDeprecated'] = true;
                    }
                    if (!$returnType && strpos($line, '@return') !== false) {
                        $returnType = trim(str_replace(['@return', '*'], '', $line));
                    }
                }
                if (!in_array($returnType, ['int', 'array', 'string', 'float'])) {
                    $returnType = 'mixed';
                }
                $methodBlade['returnType'] = $returnType;
                $methodBlade['doc'] = $methodBlade['doc'] ?: [$alias ? $alias : $methodName];
                $namespace = explode('\\', $this->aliasMap[get_class($this->classesMap[$className])]);
                array_pop($namespace);
                $dumpMap[substr($className, 0, -10)]['methods'][] = $methodBlade;
                $dumpMap[substr($className, 0, -10)]['namespace'] = implode('\\', $namespace);
                $dumpMap[substr($className, 0, -10)]['shortName'] = substr($shortClassName, 0, -10);
            }

            foreach ($dumpMap as $className => $item) {
                $path = $that->dump_path."/micro-bridge/src/Invokes/{$microName}/".str_replace('\\', '/', $item['namespace']);
                if (!is_dir($path)) {
                    mkdir($path, 0777, true);
                }
                $item['namespace'] = $item['namespace'] ? $microName.'\\'.$item['namespace'] : $microName;
                file_put_contents("$path/{$item['shortName']}.php", "<?php\n".view('bridge-controller', [
                        'microName'      => $microName,
                        'className'      => $className,
                        'shortClassName' => $item['shortName'],
                        'methods'        => $item['methods'],
                        'namespace'      => $item['namespace'],
                    ])->render());
            }
        });
        if ($this->hasError) {
            exit(1);
        }

        if ($write_json){
            file_put_contents($this->dump_path."/micro-bridge/composer.json", json_encode([
                'name'        => 'mainto/micro-bridge-'.str_replace("_", "-", snake_case($microName)),
                'homepage'    => 'https://code.hzmantu.com',
                'description' => "PHP {$microName} Serivce RPC Bridge from MainTo Company",
                'keywords'    => ["micro core bridge {$microName} sdk mainto"],
                'license'     => ['MIT'],
                'require'     => ['php' => '^7.2.1'],
                'autoload'    => [
                    'psr-4' => [
                        "Mainto\\Bridge\\Enums\\{$microName}\\"   => "src/Enums/{$microName}",
                        "Mainto\\Bridge\\Invokes\\{$microName}\\" => "src/Invokes/{$microName}",
                    ],
                ],
            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES));
        }
        return;
    }
}
