<?php
/**
 * Created by PhpStorm.
 * User: jiuli
 * Date: 2018/4/9
 * Time: 下午4:22
 */

namespace Mainto\RpcServer\RpcServer;

use Exception;
use Illuminate\Contracts\Debug\ExceptionHandler;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use InvalidArgumentException;
use Mainto\RpcServer\Exceptions\BaseNotReportServiceException;
use Mainto\RpcServer\Exceptions\RpcNotSupportException;
use Mainto\RpcServer\Exceptions\RpcStreamException;
use Mainto\RpcServer\RpcUtil\Block\ErrorReturnBlock;
use Mainto\RpcServer\RpcUtil\Block\ReturnBlock;
use Mainto\RpcServer\RpcUtil\RpcBlock;
use Mainto\RpcServer\RpcUtil\RpcGCHelper;
use Mainto\RpcServer\RpcUtil\Stream\RpcStreamAbstract;
use Mainto\RpcServer\RpcUtil\Stream\RpcStreamControlBlock;
use Mainto\RpcServer\RpcUtil\Stream\RpcStreamProviderInterface;
use Mainto\RpcServer\RpcUtil\Tool\RpcLog;
use RpcRouter;
use Throwable;
use Workerman\Connection\ConnectionInterface;

/**
 * Rpc的流传输是建立在TCP上的抽象层，该层负责双向的传输
 *
 * Class RpcStream
 * @package Mainto\RpcServer\RpcUtil\Stream
 */
class RpcStream extends RpcStreamAbstract {
    const StreamRawPayloadType = 1;

    protected static $streamMap = [];

    /**
     * 流ID
     * @var string
     */
    protected $streamId;

    /**
     * 调用者
     * @var RpcInvoke
     */
    protected $invoke;

    /**
     * @var ConnectionInterface
     */
    protected $connection;

    /**
     * 是否进入流模式
     * @var bool
     */
    protected $isStreamMode = false;

    /**
     * 当前流是否已关闭
     * @var bool
     */
    protected $isClose = false;

    /**
     * @var RpcStreamProviderInterface null
     */
    protected $streamProvider = null;

    /**
     * RpcStream constructor.
     * @param ConnectionInterface $connection
     * @param $streamId
     */
    private function __construct (ConnectionInterface $connection, $streamId) {
        $this->streamId = $streamId;
        $this->connection = $connection;
    }

    /**
     * 根据控制块获取流
     * @param ConnectionInterface $connection
     * @param string $streamId
     * @return RpcStream|null
     */
    public static function getStreamByStreamId (ConnectionInterface &$connection, string $streamId) {
        $stream = self::$streamMap[$streamId] ?? null;
        if (!$stream) {
            $stream = new RpcStream($connection, $streamId);
            self::$streamMap[$streamId] = $stream;

            $connection->streamMap[$streamId] = true;
        }

        return $stream;
    }

    /**
     * 关闭所有的流
     * @param ConnectionInterface $connection
     */
    public static function closeAllStream (ConnectionInterface &$connection) {
        if (property_exists($connection, "streamMap")) {
            foreach ($connection->streamMap as $streamId => $_) {
                if (isset(self::$streamMap[$streamId])) {
                    self::$streamMap[$streamId]->close();
                    unset(self::$streamMap[$streamId]);
                }
            }
        }
    }

    /**
     * 向流中传输一个数据
     * @param RpcStreamControlBlock $block
     * @throws Throwable
     */
    public function push (RpcStreamControlBlock $block) {
        if (RpcServer::$verboseMode) {
            Log::debug("recv {$block->getAction()} block: ".serialize($block->getPayload()));
        }

        if (!$this->isClose) {
            switch ($block->getAction()) {
                case "start":
                    $result = null;
                    $header = [];
                    $rawMode = false;
                    $source = "";
                    $start = microtime(true);

                    try {
                        $rpcBlock = RpcBlock::make($block->getPayloadType(), $block->getPayload());
                        $capture = RpcRouter::capture($this, $rpcBlock);
                        if ($capture instanceof RpcInvoke) {
                            $this->invoke = $capture;
                            /** @var RpcContext $context */
                            $context = null;
                            $source = $this->invoke->getMethodSource();
                            $result = $this->invoke->invoke($context);
                            $header = $context->getResponseHeader();
                            $rawMode = $context->isRawMode();
                        } elseif ($capture instanceof RpcBlock) {
                            $result = $capture;
                        } else {
                            throw new RpcNotSupportException("capture is not found");
                        }
                    } catch (ValidationException $e) {
                        $this->isStreamMode = false;
                        $result = new ErrorReturnBlock(422, $e->validator->errors()->getMessages());
                        // 不记录入RpcLog
                        Log::error($e);
                    } catch (InvalidArgumentException $e) {
                        $this->isStreamMode = false;
                        $result = new ErrorReturnBlock(500, $e->getMessage());
                        // 不记录入RpcLog
                        Log::error($e);
                    } catch (Throwable $e) {
                        $this->isStreamMode = false;
                        $code = intval($e->getCode());
                        $result = new ErrorReturnBlock($code ?: 500, $e->getMessage());

                        if (!($e instanceof BaseNotReportServiceException)) {
                            RpcLog::getInstance()->logThrowable($block->getPayload(), $e, $source);
                        }

                        if ($e instanceof Exception) {
                            app(ExceptionHandler::class)->report($e);
                        } else {
                            Log::error($e);
                        }
                    } finally {
                        $this->resourceRecover();
                    }

                    $header["x-runtime"] = round(microtime(true) - $start, 6)."";
                    $send = null;
                    if ($this->isStreamMode && $this->streamProvider) {
                        $payload = $header ? json_encode(['header' => $header]) : null;
                        $send = RpcStreamControlBlock::getInstanceByValue("ready", $this->streamId, $payload, self::StreamRawPayloadType);
                        $this->sendToConnection($send);
                    } else {
                        $returnBlock = $this->processResult($result, true, $rawMode);
                        if ($returnBlock instanceof ReturnBlock && $header) {
                            $returnBlock->setExtend('header', $header);
                        }
                        $this->close($returnBlock);
                    }

                    break;
                case "ready":
                    $this->streamProvider->onStreamStart($this);
                    break;
                case "stream":
                    if ($this->streamProvider == null) {
                        $this->close();
                    } else {
                        try {
                            $this->streamProvider->onMessageRecv($this, $block->getPayload());
                        } catch (Throwable $e) {
                            $this->close();
                            $this->streamProvider->onStreamClosed($block->getPayload());

                            if ($e instanceof Exception) {
                                app(ExceptionHandler::class)->report($e);
                            } else {
                                Log::error($e);
                            }
                        }
                    }
                    break;
                case "close":
                    $this->close();
                    if ($this->streamProvider) {
                        // NOT TRY: 因为关闭的内容通常都是回收内存关闭文件等操作，应该任其崩溃
                        $this->streamProvider->onStreamClosed($block->getPayload());
                    }
                    break;
            }
        } else {
            throw new RpcStreamException("stream is closed");
        }
    }

    /**
     * 回收可能导致有问题的资源
     */
    private function resourceRecover () {
        while (DB::transactionLevel()) {
            DB::rollBack();
        }
    }

    /**
     * 发送数据到客户端
     * @param RpcStreamControlBlock $block
     */
    protected function sendToConnection (RpcStreamControlBlock $block) {
        if (!$this->isClose) {
            if (RpcServer::$verboseMode) {
                Log::debug("send {$block->getAction()} block: ".serialize($block->getPayload()));
            }

            $this->connection->send($block);
        } else {
            throw new RpcStreamException("stream is closed");
        }
    }

    /**
     * 关闭连接
     * @param null $param
     */
    public function close ($param = null) {
        if (!$this->isClose) {
            $sendBlock = $this->processResult($param, !$this->isStreamMode);
            $this->sendToConnection(RpcStreamControlBlock::getInstance([
                "a" => "close",
                "i" => $this->streamId,
                "p" => $this->isStreamMode ? $sendBlock : $sendBlock->toArray(),
                "t" => $this->isStreamMode ? self::StreamRawPayloadType : $sendBlock->getPayloadType(),
            ]));

            $this->isClose = true;
            self::removeStream($this->connection, $this);

            // hit gc
            RpcGCHelper::getInstance()->hitGC();
        }
    }

    /**
     * 清空关于一个流的信息
     * @param ConnectionInterface $connection
     * @param RpcStream $stream
     */
    public static function removeStream (ConnectionInterface $connection, RpcStream $stream) {
        unset(self::$streamMap[$stream->streamId]);

        if (isset($connection->streamMap)) {
            unset($connection->streamMap[$stream->streamId]);
        }
    }

    /**
     * 打开流模式
     * @param RpcStreamProviderInterface $provider
     */
    public function setStreamMode (RpcStreamProviderInterface $provider): void {
        $this->isStreamMode = true;
        $this->streamProvider = $provider;
    }

    /**
     * 发送信息到流
     * @param $data
     */
    public function send ($data) {
        $this->sendToConnection(RpcStreamControlBlock::getInstanceByValue("stream", $this->streamId, $this->processResult($data, false), self::StreamRawPayloadType));
    }
}