<?php
/**
 * Created by PhpStorm.
 * User: jiuli
 * Date: 2018/3/16
 * Time: 上午9:49
 */

namespace Mainto\RpcServer\RpcServer;


use Doctrine\Common\Annotations\AnnotationException;
use Doctrine\Common\Annotations\AnnotationReader;
use InvalidArgumentException;
use Mainto\RpcServer\Base\Controller;
use Mainto\RpcServer\Exceptions\RpcMethodNotFoundException;
use Mainto\RpcServer\Exceptions\RpcNotSupportException;
use Mainto\RpcServer\RpcAnnotations\RpcApi;
use Mainto\RpcServer\RpcAnnotations\RpcAuthority;
use Mainto\RpcServer\RpcAnnotations\RpcCron;
use Mainto\RpcServer\RpcAnnotations\RpcDisableTypeValidation;
use Mainto\RpcServer\RpcAnnotations\RpcFormat;
use Mainto\RpcServer\RpcAnnotations\RpcHeader;
use Mainto\RpcServer\RpcAnnotations\RpcMessageHook;
use Mainto\RpcServer\RpcAnnotations\RpcParam;
use Mainto\RpcServer\RpcAnnotations\RpcResponseHeader;
use Mainto\RpcServer\RpcUtil\Block\CheckBlock;
use Mainto\RpcServer\RpcUtil\Block\CheckReturnBlock;
use Mainto\RpcServer\RpcUtil\Block\InvokeControlBlock;
use Mainto\RpcServer\RpcUtil\Block\MessageInvokeControlBlock;
use Mainto\RpcServer\RpcUtil\Block\MethodNotFoundReturnBlock;
use Mainto\RpcServer\RpcUtil\Block\UrlInvokeControlBlock;
use Mainto\RpcServer\RpcUtil\RpcBlock;
use Mainto\RpcServer\RpcUtil\RpcParameter;
use ReflectionClass;
use ReflectionException;
use ReflectionMethod;
use RuntimeException;


class RpcRouter {
    use RpcDumpTrait;

    private static $instance;

    /**
     * class Map
     * @var array
     */
    private $classesMap = [];

    /**
     * method Map
     * @var array
     */
    private $methodsMap = [];

    /**
     * params Map
     * @var array
     */
    private $paramsMap = [];

    /**
     * url Map
     * @var array
     */
    private $urlMap = [];

    /**
     * alias Map
     * @var array
     */
    private $aliasMap = [];

    /**
     * message name Map
     * @var array
     */
    private $messageNameMap = [];

    /**
     * message name to controller name Map
     * @var array
     */
    private $messageNameToControllerMap = [];

    /**
     * raw mode map
     * @var array
     */
    private $rawModeMap = [];

    /**
     * 注解解释器
     * @var AnnotationReader
     */
    private $annotationReader = null;

    /**
     * url前缀
     * @var string
     */
    private $urlPrefix = "";

    /**
     * RpcRouter constructor.
     */
    private function __construct () {
        $this->urlPrefix = config('rpc-server.url.prefix');

        try {
            $this->annotationReader = new AnnotationReader();
        } catch (AnnotationException $e) {
            dd($e);
        }
    }

    public static function getInstance () {
        if (self::$instance == null) {
            self::$instance = new RpcRouter();
        }

        return self::$instance;
    }

    /**
     * 注册一个RPC路由
     *
     * @param string $class
     * @param string|null $alias
     */
    public function register (string $class, string $alias = null) {
        try {
            $refClass = new ReflectionClass($class);
            if (!$alias) {
                $alias = $refClass->getName();
                $pos = strrpos($alias, 'Controllers\\');
                if ($pos !== false) {
                    $alias = substr($alias, $pos + 12);
                } else {
                    throw new RuntimeException($class.'not has alias and path error');
                }
            }

            $this->aliasMap[$class] = $alias;
            $this->addClass($refClass, $alias);
        } catch (ReflectionException $e) {
            /** @var RuntimeException $e */
            throw $e;
        }
    }

    /**
     * 为路由增加一个 Class 的所有公共的普通方法
     *
     * @param ReflectionClass $class
     * @param string $alias
     * @throws ReflectionException
     */
    private function addClass (ReflectionClass $class, string $alias) {
        $instance = $this->classesMap[$alias] ?? null;
        if ($instance) {
            if ($class->isInstance($instance)) {
                return;
            } else {
                $instanceClassName = (new ReflectionClass($instance))->getName();
                throw new InvalidArgumentException("class {$class->getName()} is used by {$instanceClassName}");
            }
        } else {
            $instance = $class->newInstance();
            if ($instance instanceof Controller) {
                $this->classesMap[$alias] = $instance;
            } else {
                throw new RuntimeException("register router {$class->getName()} is not support");
            }
        }

        $methods = $class->getMethods(ReflectionMethod::IS_PUBLIC);

        foreach ($methods as $method) {
            $this->addMethod($class, $method, $alias);
        }
    }

    /**
     * 为路由增加一个方法
     *
     * @param ReflectionClass $class
     * @param ReflectionMethod $method
     * @param string $classAlias
     */
    private function addMethod (ReflectionClass $class, ReflectionMethod $method, string $classAlias) {
        if (!$method->isStatic() && $method->isPublic()) {
            $className = $classAlias;
            $methodName = $method->getShortName();
            $mapKey = "{$className}::{$methodName}";

            if ($methodName[0] == "_") {
                return;
            }

            if (!isset($this->methodsMap[$mapKey])) {
                $this->methodsMap[$mapKey] = [];
            }
            $this->methodsMap[$mapKey] = $method;

            // 对参数反射进行注册
            $parameters = $method->getParameters();
            $rpcParameter = new RpcParameter();
            foreach ($parameters as $key => $parameter) {
                $rpcParameter->addParameter($parameter);
            }

            // 注册 Annotation
            $annotations = $this->annotationReader->getMethodAnnotations($method);
            $urlItem = new RpcUrlCollectItem();
            $cronItem = new RpcCronItem();
            $messageItem = new RpcMessageCollectItem();
            foreach ($annotations as $annotation) {
                if ($annotation instanceof RpcApi) {
                    $url = $annotation->url;
                    if ($annotation->addPrefix && config('rpc-server.url.enable_prefix')) {
                        $url = $this->urlPrefix.($url[0] == "/" ? $url : "/{$url}");
                    }

                    $urlKey = "[{$annotation->method}]{$url}";
                    if (isset($this->urlMap[$urlKey])) {
                        throw new RuntimeException("already has this router: {$urlKey}");
                    }
                    $this->urlMap[$urlKey] = [$className, $methodName];

                    $urlItem->setMethod($annotation->method);
                    $urlItem->setUrl($url);
                } elseif ($annotation instanceof RpcParam) {
                    if ($annotation->require) {
                        $rpcParameter->setRequired($annotation->name);
                    }

                    if ($annotation->validation) {
                        $rpcParameter->setValidation($annotation->name, $annotation->validation);
                    }

                    if ($annotation->type) {
                        $rpcParameter->setType($annotation->name, $annotation->type);
                    }

                    if ($annotation->default) {
                        $rpcParameter->setDefault($annotation->name, $annotation->default);
                    }

                    if ($annotation->nullable) {
                        $rpcParameter->setNullable($annotation->name);
                    }

                    $rpcParameter->addComment($annotation->name, $annotation->comment);
                    $rpcParameter->addDeclareName($annotation->name);
                } elseif ($annotation instanceof RpcHeader) {
                    $urlItem->addHeader($annotation->name);
                } elseif ($annotation instanceof RpcFormat) {
                    $urlItem->setFormat($annotation->format);
                    if ($annotation->format == "Raw") {
                        $this->rawModeMap[$mapKey] = true;
                    }
                } elseif ($annotation instanceof RpcResponseHeader) {
                    $urlItem->addResponseHeader($annotation->name, $annotation->value);
                } elseif ($annotation instanceof RpcAuthority) {
                    $urlItem->addAuthority($annotation->need);
                } elseif ($annotation instanceof RpcMessageHook) {
                    $messageItem->setMessageHookName($annotation->name);
                    $messageItem->setMethodName($mapKey);
                    $this->messageNameMap[$mapKey] = [$className, $methodName];
                    $this->messageNameToControllerMap[$annotation->name][] = $mapKey;
                } elseif ($annotation instanceof RpcDisableTypeValidation) {
                    if (!$annotation->check_validation) {
                        $rpcParameter->disableValidationCheck();
                    }

                    if (!$annotation->check_type) {
                        $rpcParameter->disableTypeCheck();
                    }
                } elseif ($annotation instanceof RpcCron) {
                    $cronItem->setSpec($annotation->spec, $annotation->every, $annotation->entry);
                    $cronItem->setCronHookName($mapKey);
                    $this->messageNameMap[$cronItem->getCronHookName()] = [$className, $methodName];
                    RpcCollect::addCron($cronItem);
                }
            }

            $urlItem->addResponseHeader('X-Process', 'backend/'.config('rpc-server.service_name')."/1");
            if ($urlItem->getMethod()) {
                RpcCollect::addUrl($urlItem);
            }

            if (!$messageItem->isEmpty()) {
                RpcCollect::addMessage($messageItem);
            }

            if (!$rpcParameter->isEmpty()) {
                $this->paramsMap[$mapKey] = $rpcParameter;
            }
        }
    }

    /**
     * @param RpcStream $stream
     * @param \Mainto\RpcServer\RpcUtil\RpcBlock $block
     * @return RpcInvoke|RpcBlock
     */
    public function capture ($stream, RpcBlock $block) {
        if ($block instanceof InvokeControlBlock) {
            $className = "{$block->getClass()}Controller";
            $methodName = $block->getMethod();
            $params = $block->getParams();

            return $this->getRpcInvokeByClassAndMethodName($stream, $className, $methodName, $params, $block);
        } elseif ($block instanceof UrlInvokeControlBlock) {
            $urlKey = "[{$block->getMethod()}]{$block->getUrl()}";
            if (isset($this->urlMap[$urlKey])) {
                list($className, $methodName) = $this->urlMap[$urlKey];
                $params = $block->getParams();

                return $this->getRpcInvokeByClassAndMethodName($stream, $className, $methodName, $params, $block);
            } else {
                throw new RpcMethodNotFoundException($urlKey);
            }
        } elseif ($block instanceof MessageInvokeControlBlock) {
            $messageKey = $block->getMessageName();
            if (isset($this->messageNameMap[$messageKey])) {
                list($className, $methodName) = $this->messageNameMap[$messageKey];
                $params = $block->getParams();

                return $this->getRpcInvokeByClassAndMethodName($stream, $className, $methodName, $params, $block);
            } else {
                throw new RpcMethodNotFoundException($messageKey);
            }
        } elseif ($block instanceof CheckBlock) {
            $ret = $block->getReplay();

            switch ($ret) {
                case "config":
                    $ret = json_encode(static::getBaseRouterConfig());
                    break;
            }

            return new CheckReturnBlock($ret);
        }

        throw new RpcNotSupportException("block type is not support");
    }

    /**
     * 获取控制器名称
     *
     * @param $class
     * @return string|null
     */
    public function getControllerAlias ($class) {
        return $this->aliasMap[$class] ?? null;
    }

    /**
     * 获取基础配置信息
     *
     * @return array
     */
    public static function getBaseRouterConfig () {
        $config = [
            'name'     => config('rpc-server.service_name'),
            'paths'    => RpcCollect::getUrlConfig(),
            'crons'    => [],
            'messages' => [],
        ];

        if (config('rpc-server.cron.enable')) {
            $config['crons'] = RpcCollect::getCronConfig();
        }

        if (config('rpc-server.rabbit_mq.enable')) {
            $config['messages'] = RpcCollect::getMessageConfig();
        }

        return $config;
    }

    /**
     * 根据消息名称获得监听的控制器名称
     *
     * @param $hookName
     * @return array|mixed
     */
    public function getControllerNamesByMessageHookName ($hookName) {
        return $this->messageNameToControllerMap[$hookName] ?? [];
    }

    /**
     * 根据信息获取 RpcInvoke
     * @param $stream
     * @param $className
     * @param $methodName
     * @param $params
     * @param $block
     * @return RpcInvoke|MethodNotFoundReturnBlock
     */
    private function getRpcInvokeByClassAndMethodName ($stream, $className, $methodName, &$params, &$block) {
        $mapKey = "{$className}::{$methodName}";
        $instance = $this->classesMap[$className] ?? null;
        $method = $this->methodsMap[$mapKey] ?? null;
        $paramMap = $this->paramsMap[$mapKey] ?? null;
        $rawMode = $this->rawModeMap[$mapKey] ?? false;

        if ($instance && $method) {
            return new RpcInvoke($stream, $instance, $method, $params, $paramMap, $block, $rawMode);
        } else {
            throw new RpcMethodNotFoundException($mapKey);
        }
    }
}