<?php

namespace Mainto\RpcServer\Command;

use Doctrine\Common\Annotations\AnnotationReader;
use Illuminate\Console\Command;
use Illuminate\Support\Str;
use Mainto\RpcServer\RpcAnnotations\Alias;
use Mainto\RpcServer\RpcAnnotations\RpcApi;
use Mainto\RpcServer\RpcAnnotations\RpcCallWay;
use Mainto\RpcServer\RpcAnnotations\RpcParam;
use ReflectionMethod;
use RpcRouter;

class RpcServiceApiDump extends Command {
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'rpc:dump-service-api {--dump_path=} {--base_namespace=} {--set_version=}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'RPC Service Api Dumper';

    /**
     * Dump IDE Start
     * ==========================================================
     */
    private $methodsMap = [];
    private $classesMap = [];
    /**
     * ===========================================================
     * Dump IDE End
     */
    public $dump_path;
    public $base_namespace;

    /**
     * Execute the console command.
     *
     * @return mixed
     * @throws \Doctrine\Common\Annotations\AnnotationException
     * @throws \Throwable
     */
    public function handle () {
        $dump_path = $this->option('dump_path');
        $base_namespace = $this->option('base_namespace');
        $this->dump_path = rtrim($dump_path ?: format_path('.'), '/').'/';
        $this->base_namespace = $base_namespace ?: 'App\Rpc\Controllers\\';
        $microName = studly_case(config('rpc-server.service_name', 'NoName'));

        $reader = new AnnotationReader();
        $that = &$this;
        RpcRouter::dump(function () use (&$that, $microName, $reader) {
            $dumpMap = [];
            foreach ($this->methodsMap as $name => $value) {
                /** @var ReflectionMethod $method */
                $method = $this->methodsMap[$name];
                [$className, $methodName] = explode("::", $name, 2);
                $shortClassName = $className;

                // get short class name
                if (($pos = strrpos($className, '\\')) !== false) {
                    $shortClassName = substr($className, $pos + 1);
                }
                //gen method blade

                //parse params
                $methodBlade = [];
                $methodBlade['alias'] = $methodName;
                $methodBlade['name'] = $methodName;
                $methodBlade['isDeprecated'] = false;
                $methodBlade['params'] = [];
                $continue = false;
                $hasUrl = false;
                $annotations = $reader->getMethodAnnotations($method);
                foreach ($annotations as $annotation) {
                    if ($annotation instanceof RpcCallWay && $annotation->only == 'Rpc') {
                        $continue = true;
                        break;
                    }

                    if ($annotation instanceof Alias) {
                        $methodBlade['alias'] = $annotation->name;
                    }

                    if ($annotation instanceof RpcApi) {
                        $methodBlade['url'] = $annotation->url;
                        $methodBlade['method'] = $annotation->method;
                        $hasUrl = true;
                    }

                    if ($annotation instanceof RpcParam) {
                        if ($annotation->require == true || in_array('required', explode('|', $annotation->validation))) {
                            $annotation->require = true;
                            $str = "{$annotation->type} {$annotation->name}";
                            $paramStrArr[] = $str;
                        }

                        $methodBlade['params'][] = $annotation;
                    }
                }
                if ($continue) continue;
                if (!$hasUrl) continue;

                //find deprecated and return
                $returnType = '';
                $docComment = $method->getDocComment();
                foreach (explode("\n", $docComment) as $line) {
                    if (!$methodBlade['isDeprecated'] && strpos($line, '@deprecated') !== false) {
                        $methodBlade['isDeprecated'] = true;
                    }
                    if (!$returnType && strpos($line, '@return') !== false) {
                        $returnType = trim(str_replace(['@return', '*'], '', $line));
                    }
                }
                $methodBlade['returnType'] = $returnType;
                $namespace = explode('\\', str_replace($that->base_namespace, '', get_class($this->classesMap[$className])));
                array_pop($namespace);
                $dumpMap[substr($className, 0, -10)]['methods'][] = $methodBlade;
                $dumpMap[substr($className, 0, -10)]['namespace'] = implode('/', $namespace);
                $dumpMap[substr($className, 0, -10)]['shortName'] = substr($shortClassName, 0, -10);
            }

            $service = [
                'serviceName' => $microName,
                'name'    => Str::kebab($microName),
                'version' => $that->getVersion(),
                'classes' => []
            ];

            foreach ($dumpMap as $className => $item) {
                $item['namespace'] = $item['namespace'] ? $microName.'\\'.$item['namespace'] : $microName;
                $service['classes'][] = [
                    'methods'       => $item['methods'],
                    'className'     => $className,
                    'classShotName' => $item['shortName'],
                ];
            }

            file_exists($that->dump_path) || mkdir($that->dump_path, 0777, true);
            $serviceJsonPath = $that->dump_path."service.json";
            file_put_contents($serviceJsonPath, json_encode($service, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT));
            $that->comment(">>>> write service.json {serviceName: {$microName}, version: {$that->getVersion()} ...} to {$serviceJsonPath}");
        });

        return 0;
    }

    public function getVersion () {
        $version = $this->option('set_version');

        if (!$version) {
            $version = get_git_branch_version();
        }
        return $this->parseStr($version);
    }

    private function parseStr ($str) {
        return ltrim(preg_replace('/[^\w]/m', '.', $str), 'release.');
    }
}
