<?php

namespace Mainto\RpcServer\Command;

use Doctrine\Common\Annotations\AnnotationReader;
use Illuminate\Console\Command;
use RpcRouter;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputOption;

class RpcMakeControllerTestClass extends Command {

    protected function configure () {
        parent::configure();
        $this->setName('rpc:make-controller-test')
            ->setDescription('Create a new controller test class ')
            ->addArgument('class_name', InputArgument::REQUIRED, 'The registered controller name')
            ->addOption('controller_namespace_base', 'c', InputOption::VALUE_OPTIONAL, 'The controller namespace base', 'App\Rpc\Controllers')
            ->addOption('test_class_namespace_base', 't', InputOption::VALUE_OPTIONAL, 'The test class namespace base', 'Tests\Feature\Controllers')
            ->addOption('output_dir', 'o', InputOption::VALUE_OPTIONAL, 'The test class output dir', './tests/Feature/Controllers/');
    }

    /**
     * Dump IDE Start
     * ==========================================================
     */
    private $methodsMap = [];
    private $classesMap = [];
    /**
     * ===========================================================
     * Dump IDE End
     */

    public $testNamespaceBase;
    public $controllerNamespace;

    public $testPathBase;

    /**
     * Execute the console command.
     *
     * @return mixed
     * @throws \Doctrine\Common\Annotations\AnnotationException
     */
    public function handle () {
        $class = $this->argument('class_name');
        $this->testNamespaceBase = trim($this->option('test_class_namespace_base'));
        $this->controllerNamespace = $this->option('controller_namespace_base');
        $this->testPathBase = format_path($this->option('output_dir'));
        $that = $this;
        $reader = new AnnotationReader();
        RpcRouter::dump(function () use (&$that, $reader, $class) {
            if ($class === '*') {
                $classes = $this->classesMap;
            } else {
                if (!isset($this->classesMap[$class])) {
                    $that->error("The class is not defined in rpc route! ");
                    exit;
                }
                $classes = [$this->classesMap[$class]];
            }

            foreach ($classes as $class) {
                $namespace = explode('\\', str_replace($that->controllerNamespace, '', get_class($class)));
                $unset_controller_name = array_slice($namespace, 0, count($namespace) - 1);
                $class_name = array_pop($namespace);
                $test_namespace = implode('\\', $unset_controller_name);
                $dir_name = $that->testPathBase.implode('/', $unset_controller_name);
                $file_name = $dir_name.'/'.$class_name.'Test.php';
                !file_exists($dir_name) and mkdir($dir_name, 0777, true);
                $methods = $that->getMethods(get_class($class), $this->methodsMap);
                if (!file_exists($file_name)) {
                    file_put_contents($file_name, "<?php\n".view('test-controller', [
                            'class_name'              => $class_name,
                            'controllerNamespaceBase' => $that->controllerNamespace.($test_namespace ? $test_namespace."\\".$class_name : '\\'.$class_name),
                            'methods'                 => $methods,
                            'namespace'               => $that->testNamespaceBase.($test_namespace ? $test_namespace : ''),
                            'package'                 => '@package '.$that->testNamespaceBase.($test_namespace ? $test_namespace : ''),
                        ])->render());
                } else {
                    $that->rebuildTestClassDoc($file_name, $class_name, $methods);
                }
            }
        });

        return 0;
    }

    /**
     * @param $file
     * @param $class_name
     * @param $methods
     * @throws \ReflectionException
     */
    public function rebuildTestClassDoc ($file, $class_name, $methods) {
        $testClass = $this->testNamespaceBase.str_replace(
                ['.php', realpath($this->testPathBase), '/'],
                ['', '', '\\'],
                realpath($file)
            );
        $refClass = new \ReflectionClass($testClass);
        $package = substr($refClass->name, 0, -(strlen($refClass->getShortName())) - 1);

        $doc = <<<EOF
/**
 * Class {$class_name}Test
 * @package {$package}
 *
 * --------------------------------------
 * auto gen method annotations, if the method name is already exists,
 * please use getResponse([method_name], [params])
 * --------------------------------------
EOF;
        foreach ($methods as $method) {
            $method = \Illuminate\Support\Str::ucfirst($method);
            $doc .= "\n * @method call$method(array \$params = [], bool \$enable_doc = true)";
        }
        $doc .= "\n */";

        file_put_contents($file, str_replace($refClass->getDocComment(), $doc, file_get_contents($file)));
    }

    public function getMethods ($class_name, $method_map) {
        $methods = [];
        foreach ($method_map as $item) {
            if ($item->class == $class_name) {
                $methods[] = $item->name;
            }
        }

        return $methods;
    }
}