<?php
/**
 * Created by PhpStorm.
 * User: liuzaisen
 * Date: 2018/10/18
 * Time: 8:09 PM
 */

namespace Mainto\RpcServer\Command;


use Doctrine\Common\Annotations\AnnotationReader;
use GuzzleHttp\Client;
use Illuminate\Console\Command;
use Illuminate\Support\Str;
use Mainto\RpcServer\Exceptions\BaseServiceException;
use Mainto\RpcServer\Exceptions\RpcRuntimeException;
use Mainto\RpcServer\RpcAnnotations\Alias;
use Mainto\RpcServer\TestHelper\Document\Document;
use Mainto\RpcServer\TestHelper\Document\Exception;
use ReflectionClass;
use RuntimeException;

class RpcDocUpload extends Command {
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'rpc:doc_upload {--file_path=} {--enum_dir=} {--doc_dir=} {--exception_dir=} {--set_version=} {--base_path=}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'upload document';

    /**
     * @var Document
     */
    protected $document;

    /**
     * RpcDocUpload constructor.
     */
    public function __construct () {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     * @throws \Exception
     */
    public function handle () {
        $this->initDocument();
        $this->setVersion();
        $this->setEnums();
        $this->setDoc();
        $this->setExceptions();
        $this->upload();

        return;
    }

    /**
     * 初始化文档信息
     */
    private function initDocument () {
        if (!$file_path = $this->option('file_path')) {
            $file_path = config('rpc-server.micro_doc.doc_dir').'/micro_document.json';
        }

        if (file_exists($file_path)) {
            $this->document = unserialize(file_get_contents($file_path));
        } else {
            $this->document = Document::getInstance();
        }
    }

    /**
     * 设置版本信息
     */
    private function setVersion () {
        $version = $this->option('set_version');

        if (!$version) {
            if (!$set_version = env('CI_COMMIT_REF_SLUG')) {
                if ($base_path = $this->option('base_path')) {
                    $git_file = $base_path.'/.git/HEAD';
                } else {
                    $git_file = base_path('/.git/HEAD');
                }

                if (!file_exists($git_file)) {
                    throw new \RuntimeException('get version fail');
                }

                $set_version = trim(str_replace(['ref: refs/heads/'], '', file_get_contents($git_file)));
                if (!$set_version) throw new \RuntimeException('get version fail');
            }
            $version = $set_version;
        }

        $this->document->setVersion(safe_url_string($version));
    }

    /**
     * 设置 Enum
     *
     * @throws \ReflectionException
     * @throws \Exception
     */
    private function setEnums () {
        $enum_dir = $this->option('enum_dir');
        if (!$enum_dir) $enum_dir = app_path('Enums');

        $enumFiles = get_files($enum_dir);
        foreach ($enumFiles as $enumFile) {
            if (!Str::endsWith($enumFile, 'Enum.php')) continue;

            $namespace = $this->getNamespace($enumFile);
            $_nameArr = explode('/', $enumFile);
            $className = str_replace(['/', '.php'], ['\\', ''], array_pop($_nameArr));

            $enum = $namespace.'\\'.$className;

            $reflection = new ReflectionClass($enum);
            $start = $reflection->getStartLine();
            $end = $reflection->getEndLine();
            $fp = fopen($enumFile, 'r');
            $line_count = 0;
            $content = '';
            while (($line = fgets($fp)) !== false) {
                if ($line_count >= $start - 1 && $line_count <= $end) {
                    $content .= $line;
                }
                $line_count++;
            }
            $enumClassName = $title = str_replace('App\Enums\\', '', $reflection->getName());
            $reader = new AnnotationReader();
            $annotations = $reader->getClassAnnotations($reflection);
            foreach ($annotations as $annotation) {
                if ($annotation instanceof Alias) {
                    $title = $annotation->name;
                }
            }
            $this->document->getEnums()->push([
                'class_name' => safe_url_string($enumClassName),
                'content'    => $reflection->getDocComment()."\n".$content,
                'title'      => $title,
            ]);
        }
    }

    /**
     * @param $file
     * @return string
     * @throws \Exception
     */
    private function getNamespace ($file) {
        $fp = fopen($file, 'r');
        while ($line = fgets($fp)) {
            if (strpos($line, 'namespace') !== false) {
                fclose($fp);
                return trim(str_replace(['namespace', ';'], '', $line));
            }
        }
        fclose($fp);
        throw new \Exception('not found namespace!');
    }

    /**
     * 处理文档
     */
    private function setDoc () {
        $doc_dir = $this->option('doc_dir');
        if (!$doc_dir) $doc_dir = base_path('doc');

        if (file_exists($doc_dir)) {
            $docFiles = get_files($doc_dir);
            foreach ($docFiles as $filePath) {
                if (!Str::endsWith($filePath, '.md')) continue;

                $path = str_replace($doc_dir, "", $filePath);

                $this->document->getDocs()->push([
                    'path'    => $path,
                    'content' => file_get_contents($filePath),
                ]);
            }
        }
    }

    /**
     * 设置异常
     *
     * @throws \ReflectionException
     * @throws \Exception
     */
    private function setExceptions () {
        $exceptionDir = $this->option('exception_dir');
        if (!$exceptionDir) $exceptionDir = app_path('Exceptions');

        $exceptionFiles = get_files($exceptionDir);
        $exceptionCodes = [];
        foreach ($exceptionFiles as $exceptionFile) {
            $namespace = $this->getNamespace($exceptionFile);
            $_nameArr = explode('/', $exceptionFile);
            $className = str_replace(['/', '.php'], ['\\', ''], array_pop($_nameArr));

            $exceptionClass = $namespace.'\\'.$className;
            $reflection = new ReflectionClass($exceptionClass);
            $regex = '/@method[ ]*static[ ]*([A-Z0-9a-z_]?[A-Za-z0-9_]*)[ ]*([A-Z0-9_]*)[ ]*\([ ]*\$(code|codeOrText)[ ]*=[ ]*(0x[0-9A-F]{9})[ ]*,[ ]*\$(text)[ ]*=[ ]*[\'"](.*)[\'"][ ]*\)/m';
            if ($reflection->isSubclassOf(BaseServiceException::class)) {
                $comment = '';
                $exceptionClassName = $title = str_replace('App\Exceptions\\', '', $reflection->getName());
                $reader = new AnnotationReader();
                $annotations = $reader->getClassAnnotations($reflection);
                foreach ($annotations as $annotation) {
                    if ($annotation instanceof Alias) {
                        $title = $annotation->name;
                    }
                }
                $exceptionDocument = new Exception(safe_url_string($exceptionClassName), $title);
                foreach (explode("\n", $reflection->getDocComment()) as $line) {
                    $line = trim($line, " \t\n\r\0\x0B*");

                    if (strpos($line, '@method') !== false) {
                        if (preg_match_all($regex, $line, $matches, PREG_SET_ORDER, 0)) {
                            [$_, $exceptionName, $exceptionMethodName, $firstParamName, $exceptionCode, $secondParamName, $message] = $matches[0];
                            $code = intval(substr($exceptionCode, 2), 16);
                            if (isset($exceptionCodes[$code])) {
                                throw new RpcRuntimeException("duplicate exception code on {$reflection->getName()}, code : $exceptionCode");
                            }
                            $exceptionCodes[$code] = true;

                            if ($exceptionName != $reflection->getShortName()) {
                                throw new RuntimeException("doc: {$line} return name is not equals class {$reflection->getName()}", 500);
                            }
                            $exceptionDocument->addInfo($exceptionMethodName, $code, $message, $comment ?: $message);
                            $comment = '';
                        } else {
                            throw new RpcRuntimeException("match failed on on {$reflection->getName()}, line content : $line");
                        }
                    }
                    if (strpos($line, '@exception-text') !== false) {
                        $comment = trim(str_replace(['@exception-text', '*'], '', $line));
                    }
                }

                $this->document->getExceptions()->push($exceptionDocument);
            }
        }
    }

    /**
     * 上传文档
     */
    private function upload () {
        $skip = env('UPLOAD_SKIP_CONFIRM');
        $versionText = "{$this->document->getProject()}/{$this->document->getVersion()}";
        $domain = config('rpc-server.micro_doc.doc_domain');

        if ($skip) {
            for ($i = 5; $i > 0; $i--) {
                $this->comment("\rstart upload to {$versionText} in $i seconds...");
                sleep(1);
            }
        } elseif (!$this->confirm("upload to {$versionText}?")) {
            $this->comment("cancel!");
            return;
        }

        $this->comment("uploading...");
        $client = new Client();
        $res = $client->post("{$domain}/@/api/project/version/upload", [
            'form_params' => ['data' => $this->document->toJson(JSON_UNESCAPED_UNICODE)],
        ]);

        $this->comment((string)$res->getBody());
    }
}
