<?php
/**
 * Created by PhpStorm.
 * User: jiuli
 * Date: 2018/4/2
 * Time: 上午11:09
 */

namespace Mainto\RpcServer\RpcUtil;

use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use InvalidArgumentException;
use Mainto\RpcServer\Exceptions\RpcRuntimeException;
use Mainto\RpcServer\RpcServer\RpcContext;
use Mainto\RpcServer\RpcServer\RpcDumpTrait;
use ReflectionParameter;

class RpcParameter {
    use RpcDumpTrait;

    /**
     * 是否开启类型检测
     * @var bool
     */
    private $enableTypeCheck = true;

    /**
     * 是否开启验证检测
     * @var bool
     */
    private $enableValidationCheck = true;

    /**
     * 所有声明的参数
     * @var array
     */
    private $declareName = [];

    /**
     * 反射参数内容
     * @var ReflectionParameter[]
     */
    private $parameters = [];

    /**
     * 必须包含的参数
     * @var array
     */
    private $mustHave = [];

    /**
     * 默认参数映射
     * @var array
     */
    private $defaultMap = [];

    /**
     * 校验规则映射
     * @var array
     */
    private $validationMap = [];

    /**
     * 类型Map
     * @var array
     */
    private $typeMap = [];

    /**
     * 可空的参数Map
     * @var array
     */
    private $nullableMap = [];

    /**
     * 默认参数数组
     * @var array
     */
    private $defaultArray = null;

    /**
     * 要求注入的context位置
     * @var null
     */
    private $contextPos = null;

    /**
     * 注释映射
     * @var array
     */
    private $commentMap = [];

    /**
     * 增加一个参数
     * @param ReflectionParameter $parameter
     */
    public function addParameter (ReflectionParameter $parameter) {
        $name = $parameter->getName();
        $this->parameters[] = $parameter;

        if (!$parameter->isDefaultValueAvailable()) {
            if ($parameter->getType() && $parameter->getType()->getName() == RpcContext::class) {
                if ($this->contextPos !== null) {
                    throw new InvalidArgumentException("request only have once");
                }

                $this->contextPos = count($this->parameters) - 1;
            } else {
                $this->mustHave[] = $name;
            }
        } else {
            $this->defaultMap[$name] = $parameter->getDefaultValue();
        }
    }

    /**
     * 当前参数列表是否为空
     * @return bool
     */
    public function isEmpty () {
        return count($this->parameters) == 0;
    }

    /**
     * 设置某个参数为必须
     * @param string $paramName
     */
    public function setRequired (string $paramName) {
        if (!in_array($paramName, $this->mustHave)) {
            $this->mustHave[] = $paramName;
        }
    }

    /**
     * 设置校验器
     * @param $name
     * @param $validation
     */
    public function setValidation ($name, $validation) {
        $this->validationMap[$name] = $validation;
    }

    /**
     * 设置类型
     * @param $name
     * @param $type
     */
    public function setType ($name, $type) {
        $this->typeMap[$name] = $type;
    }

    /**
     * 设置可空
     * @param $name
     * @param $type
     */
    public function setNullable ($name) {
        $this->nullableMap[$name] = true;
    }

    /**
     * 设置某个参数的默认值
     * @param string $paramName
     * @param $default
     */
    public function setDefault (string $paramName, $default) {
        $this->defaultMap[$paramName] = $default;

        if (($index = array_search($paramName, $this->mustHave)) !== false) {
            unset($this->mustHave[$index]);
        }
    }

    /**
     * 关闭类型检测
     */
    public function disableTypeCheck () {
        $this->enableTypeCheck = false;
    }

    /**
     * 关闭类型检测
     */
    public function disableValidationCheck () {
        $this->enableValidationCheck = false;
    }

    /**
     * 构造参数列表
     * @param RpcContext $request
     * @return array
     * @throws \Illuminate\Validation\ValidationException
     */
    public function buildInvokeParam (RpcContext $request) {
        $params = $this->buildParam();

        if ($this->enableTypeCheck) {
            foreach ($this->typeMap as $key => $type) {
                if ($request->has($key)) {
                    if (!$this->checkType($request, $key, $type)) {
                        throw new RpcRuntimeException("key `{$key}` is not a {$type} instance", 422);
                    }
                }
            }
        }

        foreach ($this->parameters as $key => $parameter) {
            if ($this->contextPos === $key) {
                $params[$key] = $request;
            } else {
                $name = $parameter->getName();

                if ($request->has($name)) {
                    $params[$key] = $request->input($name);
                }
            }
        }

        foreach ($this->defaultMap as $key => $value) {
            if (!$request->has($key)) {
                $request->setInput($key, $value);
            }
        }

        if ($this->enableValidationCheck && $this->validationMap) {
            $validator = Validator::make($request->all(), $this->validationMap, [], []);

            if ($validator->fails()) {
                throw new ValidationException($validator);
            }
        }

        return $params;
    }

    /**
     * 构造默认参数列表
     * @return array
     */
    private function buildParam (): array {
        if (!$this->defaultArray) {
            $defaultArray = [];
            foreach ($this->parameters as $key => $parameter) {
                $name = $parameter->getName();

                if (isset($this->defaultMap[$name])) {
                    $defaultArray[] = $this->defaultMap[$name];
                } else {
                    $defaultArray[] = null;
                }
            }

            foreach ($this->mustHave as $item) {
                if (!isset($this->validationMap[$item])) {
                    $this->validationMap[$item] = "required";
                }
            }
            $this->defaultArray = $defaultArray;
        }

        return $this->defaultArray;
    }

    /**
     * 检测类型
     *
     * @param RpcContext $request
     * @param $name
     * @param $type
     * @return bool
     */
    private function checkType (RpcContext $request, $name, $type): bool {
        $value = $request->input($name);

        // 检查可空的类型
        if ($value === null && ($this->nullableMap[$name] ?? false)) {
            return true;
        }

        switch ($type) {
            case "string":
                if (is_string($value)) {
                    $request->setInput($name, strval($value));
                    return true;
                }
                return false;
            case "uint":
                if (is_numeric($value) && $value >= 0) {
                    $request->setInput($name, intval($value));
                    return true;
                }
                return false;
            case "int":
                if (is_numeric($value)) {
                    $request->setInput($name, intval($value));
                    return true;
                }
                return false;
            case "double":
                if (is_numeric($value)) {
                    $request->setInput($name, floatval($value));
                    return true;
                }
                return false;
            case "unsignedDouble":
                if (is_numeric($value) && $value >= 0) {
                    $request->setInput($name, floatval($value));
                    return true;
                }
                return false;
            case "bool":
                if (is_bool($value)) {
                    return true;
                } else {
                    if (is_string($value) || is_numeric($value)) {
                        switch (strtolower($value)) {
                            case 'true':
                            case 1:
                                $request->setInput($name, true);
                                return true;
                            case 'false':
                            case 0:
                                $request->setInput($name, false);
                                return true;
                        }
                    }
                    return false;
                }
            case "array":
                return is_array($value);
            case "array<int>":
                if (is_array($value)) {
                    foreach ($value as $key => $val) {
                        $value[$key] = (int)$val;
                    }
                    $request->setInput($name, $value);
                    return true;
                }
                return false;
            case  "array<uint>":
                if (is_array($value)) {
                    foreach ($value as $key => $val) {
                        $val = (int)$val;
                        if ($val < 0) return false;

                        $value[$key] = (int)$val;
                    }
                    $request->setInput($name, $value);
                    return true;
                }
                return false;
            case  "array<string>":
                if (is_array($value)) {
                    foreach ($value as $key => $val) {
                        $value[$key] = (string)$val;
                    }
                    $request->setInput($name, $value);
                    return true;
                }
                return false;
            case "mixed":
                return true;
        }

        return false;
    }

    /**
     * 添加一个声明的对象名称
     * @param $name
     */
    public function addDeclareName ($name) {
        $this->declareName[] = $name;
    }

    /**
     * 添加一个声明的对象名称
     * @param $name
     * @param $value
     */
    public function addComment ($name, $value) {
        $this->commentMap[$name] = $value;
    }

    /**
     * 获得所有声明的名称
     * @return array
     */
    public function getDeclareName (): array {
        return $this->declareName;
    }
}