<?php
/**
 * Created by PhpStorm.
 * User: jiuli
 * Date: 2018/3/19
 * Time: 下午3:01
 */

namespace Mainto\RpcServer\RpcClient;


use Exception;
use Illuminate\Support\Facades\Log;
use Mainto\RpcServer\RpcServer\RpcServer;
use Mainto\RpcServer\RpcUtil\Block\ReturnBlock;
use Mainto\RpcServer\RpcUtil\RpcBlock;
use Socket\Raw\Factory;
use Throwable;

class RpcClient {
    private static $rpcConnectionMap = [];

    private $serverAddress = "";

    /**
     * Socket
     *
     * @var \Socket\Raw\Socket
     */
    private $socket;

    /**
     * RpcClient constructor.
     *
     * @param $serverAddress
     */
    private function __construct ($serverAddress) {
        $factory = new Factory();
        $this->socket = $factory->createTcp4();
        $this->socket->connectTimeout($serverAddress, 3);
        $this->socket->setBlocking(true);
        $this->serverAddress = $serverAddress;
    }

    /**
     * 根据服务名获取 Client
     *
     * @param $serverName
     * @return RpcClient|null
     */
    public static function getInstanceByServiceName ($serverName) {
        $client = get_etcd_client();
        $keys = $client->getKeysWithPrefix(RpcServer::SERVICE_PREFIX."{$serverName}/");

        if (count($keys) == 0) {
            return null;
        }
        $item = json_decode(array_random($keys), true);

        if (!isset($item['addresses']) || !isset($item['port'])) {
            return null;
        }

        foreach ($item['addresses'] as $address) {
            try {
                return self::getInstance("{$address}:{$item['port']}");
            } catch (Throwable $e) {
                Log::error("connect {$serverName}[{$address}] error: ".$e->getMessage());
            }
        }

        return null;
    }

    /**
     * 单例获取
     *
     * @param $serverAddress
     * @return RpcClient
     */
    public static function getInstance ($serverAddress) {
        if (isset(self::$rpcConnectionMap[$serverAddress])) {
            return self::$rpcConnectionMap[$serverAddress];
        } else {
            $instance = new RpcClient($serverAddress);
            self::$rpcConnectionMap[$serverAddress] = $instance;

            return $instance;
        }
    }

    public function isClosed () {
        $read = $except = $write = [$this->socket->getResource()];
        return !!socket_select($read, $write, $except, 1);
        //        foreach ($read as $conn){
        //            if (socket_read($conn, 1) === false){
        //                return false;
        //            }
        //        }
        //
        //        return true;
    }

    /**
     * 发送数据块到服务器，并将返回结果返回
     *
     * @param RpcBlock $block
     * @return ReturnBlock|RpcStreamResponse|null
     */
    public function sendBlock (RpcBlock $block) {
        $steam = new RpcStream($this->socket);

        return $steam->sendFirstBlock($block);
    }

    public function isClose () {
        try {
            $isClose = $this->socket->selectRead();
        } catch (\Socket\Raw\Exception $e) {
            return true;
        }
        return $isClose;
    }

    /**
     * 关闭客户端
     */
    public function close () {
        try {
            $this->socket->close();
        } catch (Exception $e) {
            Log::error($e);
        } finally {
            if (isset(self::$rpcConnectionMap[$this->serverAddress])) {
                unset(self::$rpcConnectionMap[$this->serverAddress]);
            }
        }
    }
}