<?php


namespace Mainto\RpcServer\RpcUtil\Tool;


use GuzzleHttp\Client;
use Illuminate\Support\Arr;
use RuntimeException;

class AgentToolHelper {
    private static $toolPort = null;

    /**
     * set config tool port
     *
     * @return null
     */
    public static function setToolPort (int $port) {
        self::$toolPort = $port;
    }

    /**
     * get config tool port
     *
     * @return null
     */
    public static function getToolPort () {
        if (!self::$toolPort) {
            return config('rpc-server.agent.tool_serve_port');
        }
        return self::$toolPort;
    }

    /**
     * get cache client
     *
     * @return Client|null
     */
    protected static function getClient () {
        static $client = null;

        if (!$client) {
            $client = new Client();
        }

        return $client;
    }

    /**
     * get setting client url
     *
     * @param string $path
     * @return string
     */
    public static function getUrl (string $path) {
        $port = AgentToolHelper::getToolPort();
        return "http://127.0.0.1:{$port}/{$path}";
    }

    /**
     * read config
     *
     * @param string $settingName
     * @return array
     */
    public static function getConfig (string $settingName) {
        $resp = self::getClient()->get(self::getUrl("setting?name={$settingName}"));

        if ($resp->getStatusCode() != 200) {
            throw new RuntimeException("can not find this setting");
        }

        $serverVersion = (int)Arr::first($resp->getHeader('version'), null, -1);
        $content = json_decode((string)$resp->getBody(), true);
        return [
            'version' => $serverVersion,
            'content' => $content,
        ];
    }

    /**
     * fast check config version
     *
     * @param string $settingName
     * @param int $version
     * @return bool
     */
    public static function checkConfigVersion (string $settingName, int $version) {
        $resp = self::getClient()->head(self::getUrl("setting?name={$settingName}"));

        if ($resp->getStatusCode() != 200) {
            throw new RuntimeException("can not find this setting");
        }

        $serverVersion = (int)Arr::first($resp->getHeader('version'), null, -1);
        return $serverVersion == $version;
    }

    /**
     * send log to agent
     *
     * @param string $logStore
     * @param array $contents
     */
    public static function sendLog (string $logStore, array $contents) {
        $resp = self::getClient()->post(self::getUrl("log/put?store={$logStore}"), [
            'form_params' => $contents,
        ]);

        $content = (string)$resp->getBody();
        if ($content != "ok") {
            throw new RuntimeException("log error: {$content}");
        }
    }

    /**
     * send rabbitmq message to agent
     *
     * @param string $exchange
     * @param string $key
     * @param bool $mandatory
     * @param bool $immediate
     * @param array $data
     */
    public static function pushToAgent (string $exchange, string $key, bool $mandatory, bool $immediate, array $data) {
        $query = http_build_query([
            'exchange'  => $exchange,
            'key'       => $key,
            'mandatory' => $mandatory ? "true" : "false",
            'immediate' => $immediate ? "true" : "false",
        ]);

        $resp = self::getClient()->post(self::getUrl("rabbitmq/put?{$query}"), [
            'form_params' => $data,
        ]);

        $content = (string)$resp->getBody();
        if ($content != "ok") {
            throw new RuntimeException("push rabbit error: {$content}");
        }
    }
}