<?php
/**
 * Created by PhpStorm.
 * User: PHPStorm
 * Date: 18-12-23
 * Time: 下午4:45
 */

namespace Mainto\RpcServer\RpcServer;


use Exception;
use Illuminate\Support\Facades\Log;
use Mainto\RpcServer\RpcUtil\RpcProtocol;
use Mainto\RpcServer\RpcUtil\RpcSocket;
use Mainto\RpcServer\RpcUtil\Stream\RpcStreamControlBlock;
use RuntimeException;
use Throwable;
use Workerman\Connection\ConnectionInterface;

class RpcServerClusterMode extends RpcServer {
    /**
     * RpcServer 实例
     * @var RpcServer
     */
    private static $instance = null;

    /**
     * @var string
     */
    private $server;

    /**
     * @var string
     */
    private $port;

    /**
     * @var string
     */
    private $key;

    /**
     * 运行
     *
     * @param string $server
     * @param string $port
     * @param string $key
     * @return mixed
     * @throws Throwable
     */
    public static function exec (string $server, string $port, string $key) {
        if (self::$instance == null) {
            self::$instance = new RpcServerClusterMode($server, $port, $key);
        }

        return self::$instance->run();
    }

    /**
     * RpcServerClusterMode constructor.
     * @param string $server
     * @param string $port
     * @param string $key
     */
    protected function __construct (string $server, string $port, string $key) {
        parent::__construct();
        $this->server = $server;
        $this->port = $port;
        $this->key = $key;
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     * @throws Throwable
     */
    public function run () {
        $buf = "";
        $blockLength = -1;
        $readerLength = 0;
        $needLength = 4;

        $socket = new RpcSocket($this->server, $this->port);
        $socket->send($this->getConnectPack($this->key));

        while (true) {
            $data = $socket->read($needLength - $readerLength);
            if ($data === false || $data === "") break;
            $readerLength += strlen($data);
            $buf = $buf.$data;

            if ($blockLength != -1 && $readerLength >= $needLength) {
                try {
                    $block = RpcProtocol::decode($buf, $socket);
                    $this->onMessage($socket, $block);

                } catch (Exception $e) {
                    Log::error($e);
                    Log::error(bin2hex($buf));
                } finally {
                    $buf = "";
                    $blockLength = -1;
                    $readerLength = 0;
                    $needLength = 4;
                }
            } else if ($blockLength == -1 && $readerLength >= 4) {
                $len = RpcProtocol::input($buf, $socket);
                if ($len === false) {
                    $socket->close();
                    throw new RuntimeException("rpc protocol report data error");
                } else if ($len === 0) {
                    continue;
                } else {
                    $blockLength = $len - 4;
                    $needLength = $len;
                }
            }
        }

        $this->onClose($socket);
        $socket->close();
        return 0;
    }

    /**
     * 当链接消息送达时
     *
     * @param ConnectionInterface $connection
     * @param RpcStreamControlBlock $block
     *
     * @throws Throwable
     */
    private function onMessage (ConnectionInterface $connection, RpcStreamControlBlock $block) {
        try {
            $steam = RpcStream::getStreamByStreamId($connection, $block->getStreamId());
            $steam->push($block);
        } catch (Throwable $e) {
            $connection->close();
            Log::error($e);
            throw $e;
        }
    }

    /**
     * 当链接被关闭时
     *
     * @param ConnectionInterface $connection
     */
    private function onClose (ConnectionInterface $connection) {
        RpcStream::closeAllStream($connection);
        if (self::$verboseMode) {
            Log::debug("Connect Close: {$connection->getRemoteAddress()}");
        }
    }

    /**
     * 获取链接包
     * @param $key
     * @return RpcStreamControlBlock
     */
    private function getConnectPack ($key) {
        return RpcStreamControlBlock::getInstance([
            "a" => "cluster-connect",
            "i" => $key,
            "t" => 0,
            "p" => [],
        ]);
    }
}