<?php

namespace Mainto\Elasticsearch;

use Elasticsearch\Client;
use Elasticsearch\ClientBuilder;
use Elasticsearch\Common\Exceptions\Missing404Exception;
use Mainto\RpcServer\Exceptions\RpcRuntimeException;
use Mainto\RpcServer\RpcUtil\Tool\RpcLog;
use Throwable;

class Service {
    /**
     * @var Client
     */
    private Client $es;

    /**
     * @var string
     */
    public string $defaultIndex;

    /**
     * @var string
     */
    public string $defaultType;

    /**
     * 构造函数
     */
    public function __construct (array $config = []) {
        // 构建es client
        $this->es = ClientBuilder::fromConfig($config);
    }

    /**
     * 设置默认索引
     * @param string $index
     * @return $this
     */
    public function setDefaultIndex (string $index): self {
        $this->defaultIndex = $index;
        return $this;
    }

    /**
     * 设置默认type
     * @param string $type
     * @return $this
     */
    public function setDefaultType (string $type): self {
        $this->defaultType = $type;
        return $this;
    }

    /**
     * 根据id快速查找
     * @param int $id
     * @param string $index
     * @param string $type
     * @return array
     */
    public function findOne (int $id, string $index = '', string $type = ''): array {
        if (!$this->defaultIndex && !$index) {
            throw new RpcRuntimeException('invalid default index');
        }

        $currentIndex = $this->defaultIndex ?? $index;

        try {
            $res = $this->es->get([
                'index' => $currentIndex,
                'id'    => $id,
            ]);
            if ($res['found'])
                return $res['_source'];
            else
                return [];
        } catch (Missing404Exception $e) {
            return [];
        } catch (Throwable $e) {
            $msg = is_string($e->getMessage()) ? json_decode($e->getMessage(), true) : $e->getMessage();
            // 可以正常捕捉到常规错误
            if (isset($msg['error']) && isset($msg['error']['reason'])) {
                throw new RpcRuntimeException($msg['error']['reason']);
            }
            throw new RpcRuntimeException('elasticsearch error :'.$e->getMessage());
        }
    }

    /**
     * 创建索引
     * @param array $indexConfig
     * @return array
     */
    public function createIndex (array $indexConfig): array {
        return $this->es->indices()->create($indexConfig);
    }

    /**
     * 删除索引
     * @param string $indexName
     * @return array
     */
    public function deleteIndex (string $indexName): array {
        $params = ['index' => $indexName];
        return $this->es->indices()->delete($params);
    }

    /**
     * 创建文档
     * @param array $params
     * @return array
     */
    public function createDocument (array $params): array {
        return $this->es->index($params);
    }

    /**
     * 删除文档
     * @param string $indexName
     * @param string $typeName
     * @param int $documentId
     * @return array
     */
    public function delDocument (string $indexName, string $typeName, int $documentId): array {
        if ($this->findOne($documentId, $indexName, $typeName)) {
            $params = [
                'index' => $indexName,
                'id'    => $documentId,
            ];
            return $this->es->delete($params);
        }
        return [];
    }

    /**
     * 根据查询删除文档
     * @param array $params
     * @return array
     */
    public function delDocumentByQuery (array $params): array {
        return $this->es->deleteByQuery($params);
    }

    /**
     * 批量创建文档
     * @param array $params
     * @return array
     */
    public function batchCreateDocument (array $params): array {
        return $this->es->bulk($params);
    }

    /**
     * 批量搜索
     * @param $param
     * @param callable|null $callback
     * @return array
     */
    public function msearch ($param, callable $callback = null) {
        try {
            $res = $this->es->msearch($param);
            if (!isset($res['responses'])) return [];
            return $callback($res['responses']);
        } catch (Throwable $e) {
            return [];
        }
    }


    /**
     * 搜索 (整理搜索结果)
     * @param $param
     * @return array
     */
    public function search ($param): array {
        try {
            $res = $this->es->search($param);
            // 查询语句没毛病
            if (isset($res['hits'])) {
                if ($res['hits']['total'] > 0) {
                    return [
                        'total' => $res['hits']['total'],
                        'data'  => $res['hits']['hits'],
                    ];
                }
                return [];
            }
            throw new RpcRuntimeException('query elasticsearch error');
        } catch (Throwable $e) {
            $msg = is_string($e->getMessage()) ? json_decode($e->getMessage(), true) : $e->getMessage();
            // 错误全部上传日志
            RpcLog::getInstance()->error('es.search.error', [
                'msg' => $msg,
            ]);
            return [];
        }
    }

    /**
     * 搜索分组后的数量
     * @param array $param
     * @return array
     */
    public function searchGroupCount (array $param): array {
        try {
            $res = $this->es->search($param);
            // 查询语句没毛病
            if (isset($res['hits'])) {
                if ($res['hits']['total'] > 0) {
                    return [
                        'total' => $res['hits']['total'],
                        'data'  => $res['aggregations'],
                    ];
                }
                return [];
            }
            throw new RpcRuntimeException('query elasticsearch error');
        } catch (Throwable $e) {
            $msg = is_string($e->getMessage()) ? json_decode($e->getMessage(), true) : $e->getMessage();
            // 错误全部上传日志
            RpcLog::getInstance()->error('es.searchGroupCount.error', [
                'msg' => $msg,
            ]);
            return [];
        }
    }

    /**
     * 原样返回搜索结果
     * @param array $param
     * @return array
     */
    public function searchRaw (array $param): array {
        return $this->es->search($param);
    }

    /**
     * 更新文档
     * @param array $param
     * @return array
     */
    public function updateDocument (array $param): array {
        return $this->es->update($param);
    }

    /**
     * 查询索引是否存在
     * @param string $indexName
     * @return bool
     */
    public function hasIndex (string $indexName): bool {
        return $this->es->indices()->exists([
            'index' => $indexName,
        ]);
    }
}