<?php

namespace Mainto\Elasticsearch;

use Mainto\RpcServer\Exceptions\RpcRuntimeException;
use Mainto\RpcServer\RpcUtil\JsonHelper;
use Mainto\RpcServer\Tracing\Span;
use Throwable;

/**
 * es帮助类  目前为单机模式
 * Class EsService
 * @package App\Service
 *
 * @mixin Service
 */
final class EsService {
    /**
     * @var EsService|null
     */
    public static ?EsService $instance = null;

    /**
     * @var Service
     */
    private Service $service;

    /**
     * @return EsService|null
     */
    public static function getInstance (): self {
        if (null == self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * 私有化实例函数
     */
    private function __construct () {
        // 构建es client
        $this->service = new Service([
            'hosts'   => [
                env('ES_HOST', 'elasticsearch-master-headless.database-elasticsearch:9200'),
            ],
            'retries' => 2,
        ]);
    }

    /**
     * 设置默认索引
     * @param string $index
     * @return $this
     */
    public function setDefaultIndex (string $index): self {
        $this->service->setDefaultIndex($index);
        return $this;
    }

    /**
     * 设置默认type
     * @param string $type
     * @return $this
     */
    public function setDefaultType (string $type): self {
        $this->service->setDefaultType($type);
        return $this;
    }

    /**
     * @param string $method
     * @param array $args
     * @return false|mixed|void|$this
     */
    public function __call (string $method, array $args) {
        $spanName = sprintf('Elasticsearch EsService::%s', $method);
        $span = Span::start($spanName);
        try {
            $span->setAttribute('es.params', JsonHelper::encodeWithUnescapedAndError($args));
            return call_user_func_array([$this->service, $method], $args);
        } catch (Throwable $e) {
            // 记录异常
            $span->statusError($e->getMessage());
            // 若为 RpcRuntimeException 错误，则向上抛出异常
            if ($e instanceof RpcRuntimeException) {
                throw $e;
            }
        } finally {
            $span->end();
        }
    }

    private function __clone () { }
}

