<?php


namespace Mainto\DDDCore\Relation;


use Mainto\DDDCore\Exception\UnknownRelationshipException;
use Mainto\DDDCore\Exception\ModelKeyIsInvalid;
use Iterator;
use Mainto\DDDCore\Interfaces\Entity;
use Mainto\DDDCore\Interfaces\Identity;

/**
 * Class ModelArray
 * @package  Mainto\DDDCore
 * @template T
 */
class ModelArray implements Iterator {
    private array $loaded = [];

    /**
     * @var $models array<T|ModelRelation>
     */
    private array $models;

    /**
     * @param array $entities
     * @return ModelArray
     */
    public static function from (array $entities): ModelArray {
        return new self($entities);
    }

    /**
     * ModelArray constructor.
     * @param array<T> $entities
     */
    public function __construct (array $entities) {
        $this->models = $entities;
    }

    /**
     * @param array<ModelRelationTrait> $modes
     * @param string $relation
     * @param Relationship $relationship
     */
    private function resolveRelationship (array $modes, string $relation, Relationship $relationship) {
        $oneResult = false;
        if ($relationship instanceof HasOne || $relationship instanceof BelongsTo) {
            $oneResult = true;
        }

        $modelMap = $localUniqueKeys = [];
        foreach ($modes as $model) {
            if ($model->loaded($relation)) {
                continue;
            }
            $localKey = getter($model, $relationship->getCurrentKey());
            if ($localKey instanceof Identity) {
                $localSerializeValue = $localKey->value();
            } elseif (is_string($localKey) || is_int($localKey)) {
                $localSerializeValue = $localKey;
            } elseif (is_null($localKey)) {
                goto init;
            } else {
                throw new ModelKeyIsInvalid();
            }

            if (!isset($modelMap[$localSerializeValue])) {
                $localUniqueKeys[] = $localKey;
            }

            $modelMap[$localSerializeValue][] = $model;
            init:
            $model->setRelation($relation, $oneResult ? null : []);
        }

        if ($localUniqueKeys) {
            $targetModels = $relationship->modelsResult($localUniqueKeys);

            if ($targetModels) {
                $targetModelGroup = [];
                foreach ($targetModels as $targetModel) {
                    $relationKey = getter($targetModel, $relationship->getTargetKey());
                    if ($relationKey instanceof Identity) {
                        $serializeValue = $relationKey->value();
                    } elseif (is_string($relationKey) || is_int($relationKey)) {
                        $serializeValue = $relationKey;
                    } else {
                        throw new ModelKeyIsInvalid();
                    }

                    $targetModelGroup[$serializeValue][] = $targetModel;
                }

                foreach ($targetModelGroup as $serializeValue => $relationModels) {
                    if (array_key_exists($serializeValue, $modelMap)) {
                        $models = $modelMap[$serializeValue];
                        foreach ($models as $model) {
                            if ($oneResult) {
                                $model->setRelation($relation, array_first($relationModels));
                            } else {
                                $model->setRelation($relation, $relationModels);
                            }
                        }
                    }
                }
            }
        }

        $this->loaded[$relation] = true;
    }

    public function load (...$relationGroups): self {
        foreach ($relationGroups as $relationGroup) {
            $relationArr = explode('.', $relationGroup);
            $prefix = "";
            $models = $this->models;
            while ($relation = array_shift($relationArr)) {
                if (!array_key_exists($relation, $this->loaded)) {
                    $first = array_first($models);
                    if (method_exists($first, $relation)) {
                        $relationship = $first->{$relation}();
                        if ($relationship instanceof Relationship) {
                            $this->resolveRelationship($models, $relation, $relationship);
                        } else {
                            throw new UnknownRelationshipException();
                        }
                    } else {
                        throw new UnknownRelationshipException();
                    }
                }
                $models = $this->pluck(ltrim($prefix.".".$relation, "."));
            }
        }

        return $this;
    }

    /**
     * @param $relation
     * @return ModelArray
     */
    public function refresh ($relation): self {
        if (array_key_exists($relation, $this->models)) {
            foreach ($this->models as $item) {
                $item->refresh($relation);
            }

            unset($this->models[$relation]);
        }

        return $this;
    }

    /**
     * @return T
     */
    public function current () {
        return current($this->models);
    }

    public function next () {
        next($this->models);
    }

    public function key () {
        return key($this->models);
    }

    public function valid (): bool {
        return key($this->models) !== null;
    }

    public function rewind () {
        reset($this->models);
    }

    public function filter (callable $callback): array {
        return array_filter($this->models, $callback);
    }

    public function map (callable $callback): array {
        $map = [];
        foreach ($this->models as $key => $value) {
            $map[] = $callback($value, $key);
        }

        return $map;
    }

    public function pluck (string $value, $key = null): array {
        $res = [];

        foreach ($this->models as $model) {
            $getValue = getter($model, $value);
            if ($key) {
                $res[getter($model, $key)] = $getValue;
            } else {
                if (is_array($getValue)) {
                    $res = array_merge($res, $getValue);
                } else {
                    $res[] = $getValue;
                }
            }
        }

        return $res;
    }

    public function each (callable $callback) {
        foreach ($this->models as $key => $value) {
            if ($callback($value, $key) === false) {
                break;
            }
        }
    }

    public function first() {
        return array_first($this->models);
    }

    /**
     * @return array|ModelRelation[]
     */
    public function values(): array {
        return $this->models;
    }

    public function unique(?string $value = null): array {
        $modelMap = $localUniqueKeys = [];

        if ($value) {
            $models = $this->pluck($value);
        } else {
            $models = $this->models;
        }

        foreach ($models as $model) {
            if ($model instanceof Identity) {
                $localSerializeValue = $model->value();
            } elseif ($model instanceof Entity) {
                $localSerializeValue = $model->identity()->value();
            }elseif (is_string($model) || is_int($model)) {
                $localSerializeValue = $model;
            } elseif (is_null($model)) {
                continue;
            } else {
                throw new ModelKeyIsInvalid();
            }

            if (!isset($modelMap[$localSerializeValue])) {
                $localUniqueKeys[] = $model;
            }

            $modelMap[$localSerializeValue][] = $model;
        }

        return $localUniqueKeys;
    }
}
