<?php

namespace Mainto\DDDCore\Command;

use Mainto\DDDCore\Command\Models\ClassRef;
use Mainto\DDDCore\Command\Models\Method;
use Mainto\DDDCore\Command\Models\Model;
use Mainto\DDDCore\Command\Models\Package;
use Mainto\DDDCore\Command\Models\Parameter;
use Mainto\DDDCore\Command\Models\Property;
use Mainto\DDDCore\Command\Models\Relationship;
use Doctrine\Common\Annotations\AnnotationReader;
use Illuminate\Console\Command;
use Illuminate\Support\Str;
use Mainto\DDDCore\Interfaces\Entity;
use Mainto\DDDCore\Interfaces\Identity;
use Mainto\DDDCore\Interfaces\ValueObject;
use Mainto\DDDCore\Relation\BelongsTo;
use Mainto\DDDCore\Relation\HasMany;
use Mainto\DDDCore\Relation\HasOne;
use Mainto\DDDCore\Relation\ModelRelationTrait;
use Mainto\RpcServer\Base\RpcEnum;
use Mainto\RpcServer\RpcAnnotations\Alias;
use Mainto\RpcServer\RpcServer\Definition\Controller;
use Mainto\RpcServer\RpcServer\Definition\NotDefinitionStruct;
use Mainto\RpcServer\RpcServer\Definition\Struct;
use Mainto\RpcServer\RpcServer\RpcDefinition;
use Mainto\RpcServer\Util\Language;
use Mainto\RpcServer\Util\ObjectMapper\ArrayTypeParser;
use ReflectionClassConstant;

class UmlCommand extends Command {
    protected $signature = 'ddd:uml {option?} {--output=}';
    private Package $topPackage;

    private array $modelIgnoreMethods;

    public function __construct () {
        parent::__construct();

        $this->topPackage = new Package("top", 0);

        $modelRelationMethods = (new \ReflectionClass(ModelRelationTrait::class))->getMethods(\ReflectionMethod::IS_PUBLIC);
        $entityMethods = (new \ReflectionClass(Entity::class))->getMethods(\ReflectionMethod::IS_PUBLIC);
        $identityMethods = (new \ReflectionClass(Identity::class))->getMethods(\ReflectionMethod::IS_PUBLIC);
        $valueObjectMethods = (new \ReflectionClass(ValueObject::class))->getMethods(\ReflectionMethod::IS_PUBLIC);
        $this->modelIgnoreMethods = array_merge([
            'jsonSerialize', 'toArray',
        ], array_map(function (\ReflectionMethod $reflectionMethod) {
            return $reflectionMethod->getName();
        }, array_merge($modelRelationMethods, $entityMethods, $identityMethods, $valueObjectMethods)));
    }

    public function resolveEnum ($class) {
        $reflectionClass = new \ReflectionClass($class);

        $currentName = preg_replace('/.*[aA]pp\\\\[\w]+\\\\?/', '', $reflectionClass->getNamespaceName());
        $currentPackage = $this->topPackage;
        foreach (explode('\\', trim($currentName, '\\')) as $namespace) {
            $currentPackage = $currentPackage->firstOrNewPackage($namespace);
        }

        $enumModel = new Model("class");
        $alias = app(AnnotationReader::class)->getClassAnnotation($reflectionClass, Alias::class);
        if ($alias) {
            $enumModel->comment = $alias->name;
        }

        $enumModel->name = class_basename($reflectionClass->getName());
        $enumModel->properties = array_map(function (ReflectionClassConstant $constant) {
            $value = new Property();
            $value->comment = get_doc($constant->getDocComment());
            $value->default = $constant->getValue();
            $value->hasDefault = true;
            $value->name = $constant->getName();
            $value->type = is_string($constant->getValue()) ? 'string' : 'int';

            return $value;
        }, array_filter($reflectionClass->getReflectionConstants(), function (ReflectionClassConstant $constant) {
            return is_string($constant->getValue()) || is_int($constant->getValue());
        }));

        $currentPackage->models[] = $enumModel;
    }

    public function handle () {
        $option = value_if($this->argument('option'), 'all');
        if ($option == 'domain') {
            $this->readerDomain();
        } elseif ($option == "interface") {
            $this->readerInterface();
        } elseif ($option == "all") {
            $this->readerDomain();
            $this->readerInterface();
        }
    }

    public function readerInterface () {
        RpcDefinition::getInstance()->controllers->each(function (Controller $controller) {
            $model = new Model("class");
            $model->name = $controller->name;
            $currentPackage = $this->topPackage;
            $namespaceArr = explode('\\', trim($controller->registerName, '\\'));
            foreach (array_splice($namespaceArr, 0, count($namespaceArr) - 1) as $namespace) {
                $currentPackage = $currentPackage->firstOrNewPackage($namespace);
            }
            foreach ($controller->methods as $method) {
                $modelMethod = new Method();
                $modelMethod->name = $method->name;
                if ($method->alias != $method->name) {
                    $modelMethod->comment = $method->alias;
                }
                [$type, $childType] = $this->parserTypeString($method->responseType);
                $modelMethod->returnType = $type;

                if (Str::endsWith($modelMethod->returnType, '[]') && !in_array($childType, Language::$simpleType)) {
                    $modelMethod->returnRelationType = "HasMany";
                    $modelMethod->returnTargetType = class_basename($childType);
                }

                if (!Str::endsWith($modelMethod->returnType, '[]')
                    && !in_array($childType, Language::$simpleReturnTypeMap)
                ) {
                    $modelMethod->returnRelationType = "HasOne";
                    $modelMethod->returnTargetType = class_basename($childType);
                }

                $model->methods[] = $modelMethod;
            }

            $currentPackage->models[] = $model;
        });

        RpcDefinition::getInstance()->structs->each(function (Struct $struct) {
            if ($struct->name == RpcDefinition::EmptyType) return;

            if ($struct instanceof NotDefinitionStruct) return;

            $currentName = preg_replace('/.*[aA]pp\\\\[\w]+\\\\?/', '', $struct->namespace);
            $currentName = preg_replace('/\\\\?Interface\\\\?/', '\\', $currentName);
            $currentPackage = $this->topPackage;
            foreach (explode('\\', trim($currentName, '\\')) as $namespace) {
                $currentPackage = $currentPackage->firstOrNewPackage($namespace);
            }

            $model = new Model('class');
            $model->name = $struct->name;
            foreach ($struct->getProperties() as $property) {
                $modelProperty = new Property();
                $modelProperty->name = $property->name;
                $modelProperty->comment = $property->comment;
                $modelProperty->hasDefault = $property->defaultValueAvailable;
                $modelProperty->default = $property->default;
                $modelProperty->type = $property->type;
                $model->properties[] = $modelProperty;
            }

            $currentPackage->models[] = $model;
        });

        $uml = "```plantuml\n@startuml test\n";
        $uml .= $this->render($this->topPackage);
        $uml .= "@enduml\n```";

        file_put_contents(value_if($this->option('output'), app_path('../doc/ddd_interface.md')), $uml);
    }

    public function readerDomain () {
        $appDir = app_path();
        $classes = get_classes($appDir);
        foreach ($classes as $class) {
            if (in_array(Entity::class, class_implements($class))) {
                $this->resolveEntity($class);
            } elseif (is_subclass_of($class, RpcEnum::class)) {
                $this->resolveEnum($class);
            } elseif (in_array(ValueObject::class, class_implements($class)) && !in_array(Identity::class, class_implements($class))) {
                $this->resolveVO($class);
            }
        }

        $uml = "```plantuml\n@startuml test\n";
        $uml .= $this->render($this->topPackage);
        $uml .= "@enduml\n```";

        file_put_contents(value_if($this->option('output'), app_path('../doc/ddd_domain.md')), $uml);
    }

    private function render (Package $package): string {
        $uml = "";

        if ($package->level == 0) {
            foreach ($package->getPackages() as $childPackage) {
                $uml .= $this->render($childPackage);
            }
            return $uml;
        }

        if ($package->models || $package->getPackages()->toArray()) {
            $uml .= str_repeat("    ", $package->level)."package {$package->getPackageName()} {\n";
        } else {
            return "";
        }
        foreach ($package->models as $model) {
            if ($model->comment) {
                $uml .= str_repeat("    ", $package->level + 1)."note top of {$model->name} : {$model->name}\n";
            }
            if ($model->isAbstract) {
                $uml .= str_repeat("    ", $package->level + 1)."abstract {$model->modelType} {$model->name} {\n";
            } else {
                $uml .= str_repeat("    ", $package->level + 1)."{$model->modelType} {$model->name} {\n";
            }

            if ($model->relationships) {
                $uml .= str_repeat("    ", $package->level + 2).".. relationships ..\n";
                foreach ($model->relationships as $relationship) {
                    $uml .= str_repeat("    ", $package->level + 2)."+ $relationship->targetType $relationship->name\n";
                }
            }

            if ($model->properties) {
                $uml .= str_repeat("    ", $package->level + 2).".. properties ..\n";
                foreach ($model->properties as $property) {
                    if ($property->comment) {
                        $uml .= str_repeat("    ", $package->level + 2)."// $property->comment\n";
                    }
                    $uml .= str_repeat("    ", $package->level + 2)."+ ".class_basename($property->type)." $property->name";
                    if ($property->hasDefault) {
                        $uml .= " = " .var_export_min($property->default);
                    }
                    $uml .= str_repeat("    ", $package->level + 2)."\n";
                }
            }
            if ($model->methods) {
                $uml .= str_repeat("    ", $package->level + 2).".. methods ..\n";
                foreach ($model->methods as $method) {
                    if ($method->comment) {
                        $uml .= str_repeat("    ", $package->level + 2)."// $method->comment\n";
                    }

                    $uml .= str_repeat("    ", $package->level + 2)."+ {$method->name}(): $method->returnType\n";
                }
            }
            $uml .= str_repeat("    ", $package->level + 1)."}\n";

            foreach ($model->relationships as $relationship) {
                if ($relationship->relationType == "HasOne") {
                    $uml .= str_repeat("    ", $package->level + 1)."$model->name::$relationship->name \"1\" --> \"1\" $relationship->targetType\n";
                } elseif ($relationship->relationType == "HasMany") {
                    $uml .= str_repeat("    ", $package->level + 1)."$model->name::$relationship->name \"1\" --> \"n\" $relationship->targetType\n";
                } elseif ($relationship->relationType == "BelongsTo") {
                    $uml .= str_repeat("    ", $package->level + 1)."$model->name::$relationship->name \"1\" <-- \"1\" $relationship->targetType\n";
                }
            }
            if ($model->parent) {
                $uml .= str_repeat("    ", $package->level + 1)."$model->name --|> $model->parent\n";
            }


        }
        foreach ($package->getPackages() as $childPackage) {
            $uml .= $this->render($childPackage);
        }

        foreach ($package->models as $model) {
            foreach ($model->methods as $method) {
                if ($method->returnRelationType == "HasOne") {
                    $uml .= str_repeat("    ", $package->level + 1)."$model->name::$method->name \"1\" --> \"1\" $method->returnTargetType\n";
                } elseif ($method->returnRelationType == "HasMany") {
                    $uml .= str_repeat("    ", $package->level + 1)."$model->name::$method->name \"1\" --> \"n\" $method->returnTargetType\n";
                } elseif ($method->returnRelationType == "BelongsTo") {
                    $uml .= str_repeat("    ", $package->level + 1)."$model->name::$method->name \"1\" <-- \"1\" $method->returnTargetType\n";
                }
            }
        }

        $uml .= "\n ".str_repeat("    ", $package->level)."}\n";

        return $uml;
    }

    private function resolveVO ($class) {
        $reflectionClass = new \ReflectionClass($class);
        $currentName = preg_replace('/.*[aA]pp\\\\[\w]+\\\\?/', '', $reflectionClass->getNamespaceName());
        $currentName = preg_replace('/\\\\?Model\\\\?/', '\\', $currentName);
        $currentPackage = $this->topPackage;
        foreach (explode('\\', trim($currentName, '\\')) as $namespace) {
            $currentPackage = $currentPackage->firstOrNewPackage($namespace);
        }

        $valueObject = new Model("class");
        $valueObject->name = $reflectionClass->getShortName();
        $alias = app(AnnotationReader::class)->getClassAnnotation($reflectionClass, Alias::class);
        if ($alias) {
            $valueObject->comment = $alias->name;
        }
        if ($reflectionClass->isAbstract()) {
            $valueObject->isAbstract = true;
        }

        $parent = $reflectionClass->getParentClass();
        if ($parent) {
            $valueObject->parent = class_basename($parent->getName());
        }

        $defaultProperties = $reflectionClass->getDefaultProperties();

        foreach ($reflectionClass->getProperties(\ReflectionProperty::IS_PUBLIC) as $reflectionProperty) {
            $property = new Property();
            $property->name = $reflectionProperty->name;
            $type = $reflectionProperty->getType();
            if ($type) {
                $property->type = class_basename($type->getName());
            }
            if (array_key_exists($reflectionProperty->getName(), $defaultProperties)) {
                $property->default = $defaultProperties[$reflectionProperty->getName()];
                $property->hasDefault = true;
            }
            $valueObject->properties[] = $property;
        }

        foreach ($reflectionClass->getMethods(\ReflectionMethod::IS_PUBLIC) as $reflectionMethod) {
            if (Str::startsWith($reflectionMethod->getName(), "__") || in_array($reflectionMethod->getName(), $this->modelIgnoreMethods)) {
                continue;
            }

            if ($reflectionMethod->getReturnType() && in_array($reflectionMethod->getReturnType()->getName(), [
                    HasOne::class,
                    HasMany::class,
                    BelongsTo::class,
                ])) {
                $relationship = new Relationship();
                $relationship->name = $reflectionMethod->name;
                $instance = $reflectionMethod->invoke($reflectionClass->newInstanceWithoutConstructor());

                if ($instance instanceof HasOne) {
                    $relationCall = $instance->getProvider()->modelToOne();
                } elseif ($instance instanceof HasMany) {
                    $relationCall = $instance->getProvider()->modelToMany();
                } elseif ($instance instanceof BelongsTo) {
                    $relationCall = $instance->getProvider()->modelToOne();
                } else {
                    throw new \RuntimeException("fail relation");
                }
                $relationProviderMethod = new \ReflectionMethod($relationCall[0], $relationCall[1]);
                $relationship->targetType = class_basename($relationProviderMethod->getReturnType()->getName());
                $relationship->relationType = class_basename($reflectionMethod->getReturnType()->getName());

                $valueObject->relationships[] = $relationship;
                continue;
            }

            $method = new Method();
            $method->name = $reflectionMethod->getName();
            foreach ($reflectionMethod->getParameters() as $reflectionParameter) {
                $parameter = new Parameter();
                $parameter->name = $reflectionParameter->getName();
                if ($reflectionParameter->getType()) {
                    $parameter->type = $reflectionParameter->getType()->getName();
                }

                if ($reflectionParameter->isDefaultValueAvailable()) {
                    $parameter->default = $reflectionParameter->getDefaultValue();
                }

                $method->parameters[] = $parameter;
            }

            if ($reflectionMethod->getReturnType()) {
                $method->returnType = class_basename($reflectionMethod->getReturnType()->getName());
            }
            $valueObject->methods[] = $method;
        }

        $currentPackage->models[] = $valueObject;
    }

    private function resolveEntity ($class) {
        $reflectionClass = new \ReflectionClass($class);

        $currentName = preg_replace('/.*[aA]pp\\\\[\w]+\\\\?/', '', $reflectionClass->getNamespaceName());
        $currentName = preg_replace('/\\\\?Model\\\\?/', '\\', $currentName);
        $currentPackage = $this->topPackage;
        foreach (explode('\\', trim($currentName, '\\')) as $namespace) {
            $currentPackage = $currentPackage->firstOrNewPackage($namespace);
        }
        $entity = new Model("entity");
        $entity->name = $reflectionClass->getShortName();
        $alias = app(AnnotationReader::class)->getClassAnnotation($reflectionClass, Alias::class);
        if ($alias) {
            $entity->comment = $alias->name;
        }

        $classRef = new ClassRef($class);
        $entity->properties = $classRef->getProperties();
        foreach ($reflectionClass->getMethods(\ReflectionMethod::IS_PUBLIC) as $reflectionMethod) {
            if (Str::startsWith($reflectionMethod->getName(), "__") || in_array($reflectionMethod->getName(), $this->modelIgnoreMethods)) {
                continue;
            }

            if ($reflectionMethod->getReturnType() && in_array($reflectionMethod->getReturnType()->getName(), [
                    HasOne::class,
                    HasMany::class,
                    BelongsTo::class,
                ])) {
                $relationship = new Relationship();
                $relationship->name = $reflectionMethod->name;
                $instance = $reflectionMethod->invoke($reflectionClass->newInstanceWithoutConstructor());

                if ($instance instanceof HasOne) {
                    $relationCall = $instance->getProvider()->modelToOne();
                } elseif ($instance instanceof HasMany) {
                    $relationCall = $instance->getProvider()->modelToMany();
                } elseif ($instance instanceof BelongsTo) {
                    $relationCall = $instance->getProvider()->modelToOne();
                } else {
                    throw new \RuntimeException("fail relation");
                }
                $relationProviderMethod = new \ReflectionMethod($relationCall[0], $relationCall[1]);
                $relationship->targetType = class_basename($relationProviderMethod->getReturnType()->getName());
                $relationship->relationType = class_basename($reflectionMethod->getReturnType()->getName());

                $entity->relationships[] = $relationship;
                continue;
            }

            $method = new Method();
            $method->name = $reflectionMethod->getName();
            foreach ($reflectionMethod->getParameters() as $reflectionParameter) {
                $parameter = new Parameter();
                $parameter->name = $reflectionParameter->getName();
                if ($reflectionParameter->getType()) {
                    $parameter->type = $reflectionParameter->getType()->getName();
                }

                if ($reflectionParameter->isDefaultValueAvailable()) {
                    $parameter->default = $reflectionParameter->getDefaultValue();
                }

                $method->parameters[] = $parameter;
            }

            if ($reflectionMethod->getReturnType()) {
                $method->returnType = class_basename($reflectionMethod->getReturnType()->getName());
            }
            $entity->methods[] = $method;
        }

        $currentPackage->models[] = $entity;
    }


    private function parserTypeString ($type): array {
        $childType = $type;
        if (Str::startsWith($type, "Map<")) {
            preg_match('/Map<(.*)>/', $type, $match);
            $explode = explode(',', $match[1]);
            $childType = ltrim($explode[1]);
            $type = "Map<$explode[0], ".class_basename($childType).">";
        } else if (Str::endsWith($type, "[]")) {
            [$childType, $arrayDeepCount] = ArrayTypeParser::getArrayType($type);
            $type = class_basename($childType) . str_repeat('[]', $arrayDeepCount);
        } else {
            $type = class_basename($type);
        }

        return [$type, $childType];
    }

}
