<?php

namespace Mainto\DDDCore\Command;

use Mainto\DDDCore\Command\Models\ClassRef;
use Mainto\DDDCore\Command\Models\Enum;
use Mainto\DDDCore\Command\Models\EnumValue;
use Mainto\DDDCore\Command\Models\Method;
use Mainto\DDDCore\Command\Models\Parameter;
use Mainto\DDDCore\Command\Models\Property;
use Mainto\DDDCore\Command\Models\Relationship;
use Mainto\DDDCore\Command\Models\Uml;
use Doctrine\Common\Annotations\AnnotationReader;
use Illuminate\Console\Command;
use Illuminate\Support\Str;
use Mainto\DDDCore\Interfaces\Entity;
use Mainto\DDDCore\Interfaces\Identity;
use Mainto\DDDCore\Relation\BelongsTo;
use Mainto\DDDCore\Relation\HasMany;
use Mainto\DDDCore\Relation\HasOne;
use Mainto\DDDCore\Relation\ModelRelationTrait;
use Mainto\RpcServer\Base\RpcEnum;
use Mainto\RpcServer\RpcAnnotations\Alias;
use Mainto\RpcServer\Util\Language;
use Mainto\RpcServer\Util\ObjectMapper\ArrayTypeParser;
use ReflectionClassConstant;

class UmlCommand extends Command {
    protected $signature = 'ddd:uml {--output=}';
    private Uml $uml;

    private array $modelIgnoreMethods;

    public function __construct () {
        parent::__construct();

        $this->uml = new Uml();

        $modelRelationMethods = (new \ReflectionClass(ModelRelationTrait::class))->getMethods(\ReflectionMethod::IS_PUBLIC);
        $entityMethods = (new \ReflectionClass(Entity::class))->getMethods(\ReflectionMethod::IS_PUBLIC);
        $identityMethods = (new \ReflectionClass(Identity::class))->getMethods(\ReflectionMethod::IS_PUBLIC);
        $valueObjectMethods = (new \ReflectionClass(\Mainto\DDDCore\Interfaces\ValueObject::class))->getMethods(\ReflectionMethod::IS_PUBLIC);
        $this->modelIgnoreMethods = array_map(function (\ReflectionMethod $reflectionMethod) {
            return $reflectionMethod->getName();
        }, array_merge($modelRelationMethods, $entityMethods, $identityMethods, $valueObjectMethods));
    }

    public function resolveEnum ($class) {
        $reflectionClass = new \ReflectionClass($class);

        $currentName = preg_replace('/.*[aA]pp\\\\[\w]+\\\\?/', '', $reflectionClass->getNamespaceName());
        $currentName = preg_replace('/\\\\?Enum\\\\?/', '\\', $currentName);
        $simpleNamespace = trim($currentName, '\\');
        $package = $this->uml->findOrNewPackage($simpleNamespace);

        $enumModel = new Enum();
        $alias = app(AnnotationReader::class)->getClassAnnotation($reflectionClass, Alias::class);
        if ($alias) {
            $enumModel->comment = $alias->name;
        }

        $enumModel->name = class_basename($reflectionClass->getName());
        $enumModel->values = array_map(function (ReflectionClassConstant $constant) {
            $value = new EnumValue();
            $value->comment = get_doc($constant->getDocComment());
            $value->value = $constant->getValue();
            $value->name = $constant->getName();

            return $value;
        }, array_filter($reflectionClass->getReflectionConstants(), function (ReflectionClassConstant $constant) {
            return is_string($constant->getValue()) || is_int($constant->getValue());
        }));

        $package->enums[] = $enumModel;
    }

    public function handle () {
        $appDir = app_path();
        $classes = get_classes($appDir);
        foreach ($classes as $class) {
            if (in_array(Entity::class, class_implements($class))) {
                $this->resolveEntity($class);
            } elseif (is_subclass_of($class, RpcEnum::class)) {
                $this->resolveEnum($class);
            } elseif (in_array(\Mainto\DDDCore\Interfaces\ValueObject::class, class_implements($class)) && !in_array(Identity::class, class_implements($class))) {
                $this->resolveVO($class);
            }
        }

        file_put_contents(value_if($this->option('output'), storage_path('ddd.puml')), view('uml', [
            'packages' => $this->uml->getPackages(),
        ])->render());
    }


    private function resolveVO($class) {
        $reflectionClass = new \ReflectionClass($class);
        $currentName = preg_replace('/.*[aA]pp\\\\[\w]+\\\\?/', '', $reflectionClass->getNamespaceName());
        $currentName = preg_replace('/\\\\?Model\\\\VO\\\\?/', '\\', $currentName);
        $simpleNamespace = trim($currentName, '\\');

        $valueObject = new Models\ValueObject();
        $valueObject->name = $reflectionClass->getShortName();
        $alias = app(AnnotationReader::class)->getClassAnnotation($reflectionClass, Alias::class);
        if ($alias) {
            $valueObject->comment = $alias->name;
        }
        if ($reflectionClass->isAbstract()) {
            $valueObject->isAbstract = true;
        }

        $parent = $reflectionClass->getParentClass();
        if ($parent) {
            $valueObject->parent = class_basename($parent->getName());
        }

        $defaultProperties = $reflectionClass->getDefaultProperties();

        foreach ($reflectionClass->getProperties(\ReflectionProperty::IS_PUBLIC) as $reflectionProperty) {
            $property = new Property();
            $property->name = $reflectionProperty->name;
            $type = $reflectionProperty->getType();
            if ($type) {
                $property->type = class_basename($type->getName());
            }
            if (array_key_exists($reflectionProperty->getName(), $defaultProperties)) {
                $property->default = $defaultProperties[$reflectionProperty->getName()];
                $property->hasDefault = true;
            }
            $valueObject->properties[] = $property;
        }

        foreach ($reflectionClass->getMethods(\ReflectionMethod::IS_PUBLIC) as $reflectionMethod) {
            if (Str::startsWith($reflectionMethod->getName(), "__") || in_array($reflectionMethod->getName(), $this->modelIgnoreMethods)) {
                continue;
            }

            if ($reflectionMethod->getReturnType() && in_array($reflectionMethod->getReturnType()->getName(), [
                    HasOne::class,
                    HasMany::class,
                    BelongsTo::class,
                ])) {
                $relationship = new Relationship();
                $relationship->name = $reflectionMethod->name;
                $instance = $reflectionMethod->invoke($reflectionClass->newInstanceWithoutConstructor());

                if ($instance instanceof HasOne) {
                    $relationCall = $instance->getProvider()->modelToOne();
                } elseif ($instance instanceof HasMany) {
                    $relationCall = $instance->getProvider()->modelToMany();
                } elseif ($instance instanceof BelongsTo) {
                    $relationCall = $instance->getProvider()->modelToOne();
                } else {
                    throw new \RuntimeException("fail relation");
                }
                $relationProviderMethod = new \ReflectionMethod($relationCall[0], $relationCall[1]);
                $relationship->targetType = class_basename($relationProviderMethod->getReturnType()->getName());
                $relationship->relationType = class_basename($reflectionMethod->getReturnType()->getName());

                $valueObject->relationships[] = $relationship;
                continue;
            }

            $method = new Method();
            $method->name = $reflectionMethod->getName();
            foreach ($reflectionMethod->getParameters() as $reflectionParameter) {
                $parameter = new Parameter();
                $parameter->name = $reflectionParameter->getName();
                if ($reflectionParameter->getType()) {
                    $parameter->type = $reflectionParameter->getType()->getName();
                }

                if ($reflectionParameter->isDefaultValueAvailable()) {
                    $parameter->default = $reflectionParameter->getDefaultValue();
                }

                $method->parameters[] = $parameter;
            }

            if ($reflectionMethod->getReturnType()) {
                $method->returnType = class_basename($reflectionMethod->getReturnType()->getName());
            }
            $valueObject->methods[] = $method;
        }

        $package = $this->uml->findOrNewPackage($simpleNamespace);
        $package->valueObjects[] = $valueObject;
    }

    private function resolveEntity ($class) {
        $reflectionClass = new \ReflectionClass($class);

        $currentName = preg_replace('/.*[aA]pp\\\\[\w]+\\\\?/', '', $reflectionClass->getNamespaceName());
        $currentName = preg_replace('/\\\\?Model\\\\Entity\\\\?/', '\\', $currentName);
        $simpleNamespace = trim($currentName, '\\');
        $entity = new Models\Entity();
        $entity->name = $reflectionClass->getShortName();
        $alias = app(AnnotationReader::class)->getClassAnnotation($reflectionClass, Alias::class);
        if ($alias) {
            $entity->comment = $alias->name;
        }

        $classRef = new ClassRef($class);
        $entity->properties = $classRef->getProperties();
        foreach ($reflectionClass->getMethods(\ReflectionMethod::IS_PUBLIC) as $reflectionMethod) {
            if (Str::startsWith($reflectionMethod->getName(), "__") || in_array($reflectionMethod->getName(), $this->modelIgnoreMethods)) {
                continue;
            }

            if ($reflectionMethod->getReturnType() && in_array($reflectionMethod->getReturnType()->getName(), [
                    HasOne::class,
                    HasMany::class,
                    BelongsTo::class,
                ])) {
                $relationship = new Relationship();
                $relationship->name = $reflectionMethod->name;
                $instance = $reflectionMethod->invoke($reflectionClass->newInstanceWithoutConstructor());

                if ($instance instanceof HasOne) {
                    $relationCall = $instance->getProvider()->modelToOne();
                } elseif ($instance instanceof HasMany) {
                    $relationCall = $instance->getProvider()->modelToMany();
                } elseif ($instance instanceof BelongsTo) {
                    $relationCall = $instance->getProvider()->modelToOne();
                } else {
                    throw new \RuntimeException("fail relation");
                }
                $relationProviderMethod = new \ReflectionMethod($relationCall[0], $relationCall[1]);
                $relationship->targetType = class_basename($relationProviderMethod->getReturnType()->getName());
                $relationship->relationType = class_basename($reflectionMethod->getReturnType()->getName());

                $entity->relationships[] = $relationship;
                continue;
            }

            $method = new Method();
            $method->name = $reflectionMethod->getName();
            foreach ($reflectionMethod->getParameters() as $reflectionParameter) {
                $parameter = new Parameter();
                $parameter->name = $reflectionParameter->getName();
                if ($reflectionParameter->getType()) {
                    $parameter->type = $reflectionParameter->getType()->getName();
                }

                if ($reflectionParameter->isDefaultValueAvailable()) {
                    $parameter->default = $reflectionParameter->getDefaultValue();
                }

                $method->parameters[] = $parameter;
            }

            if ($reflectionMethod->getReturnType()) {
                $method->returnType = class_basename($reflectionMethod->getReturnType()->getName());
            }
            $entity->methods[] = $method;
        }

        $package = $this->uml->findOrNewPackage($simpleNamespace);
        $package->entities[] = $entity;
    }
}
