<?php

use Illuminate\Support\Str;

if (!function_exists('get_classes')) {
    function get_classes ($dir): array {
        $files = get_files($dir);
        $classes = [];
        foreach ($files as $file) {
            if (Str::endsWith($file, '.php')) {
                $class = get_class_from_file($file);
                if ($class !== null) {
                    $classes[] = $class;
                }
            }
        }

        return $classes;
    }
}

if (!function_exists('get_class_from_file')) {
    function get_class_from_file ($phpFile): ?string {
        $content = file_get_contents($phpFile);
        $tokens = token_get_all($content);
        $namespace = '';
        for ($index = 0; isset($tokens[$index]); $index++) {
            if (!isset($tokens[$index][0])) {
                continue;
            }
            if (T_NAMESPACE === $tokens[$index][0]) {
                $index += 2; // Skip namespace keyword and whitespace
                while (isset($tokens[$index]) && is_array($tokens[$index])) {
                    $namespace .= $tokens[$index++][1];
                }
            }
            if (T_CLASS === $tokens[$index][0] && T_WHITESPACE === $tokens[$index + 1][0] && T_STRING === $tokens[$index + 2][0]) {
                $index += 2; // Skip class keyword and whitespace
                return $namespace.'\\'.$tokens[$index][1];

                # break if you have one class per file (psr-4 compliant)
                # otherwise you'll need to handle class constants (Foo::class)
            }
        }

        return null;
    }
}

if (!function_exists('get_files')) {
    /**
     * Get files
     * @param $dir
     * @return array
     */
    function get_files ($dir): array {
        $files = [];
        $scan = scandir($dir);
        foreach ($scan as $item) {
            if ($item == '.' || $item == '..') continue;
            if (is_dir($dir.'/'.$item)) {
                $files = array_merge($files, get_files($dir.'/'.$item));
            } else {
                $files[] = $dir.'/'.$item;
            }
        }
        return $files;
    }
}

if (!function_exists('value_if')) {
    function value_if ($first, $default) {
        return $first ?: $default;
    }
}

if (!function_exists('getter')) {
    /**
     * @param object $object
     * @param string $property
     * @return object|mixed
     */
    function getter(object $object, string $property) {
        $value = $object;

        $propertyArr = explode('.', $property);
        while ($key = array_shift($propertyArr)) {
            $getMethod = 'get'.Str::studly($key);
            if (method_exists($value, $getMethod)) {
                $value = $value->{$getMethod}();
            } else {
                $value = $value->{$key};
            }
        }

        return $value;
    }
}


if (!function_exists('var_export_min')) {
    /**
     * dump var
     *
     * @param $var
     * @param bool $return
     * @return mixed|string
     */
    function var_export_min ($var, $return = true): string {
        if (is_array($var)) {
            $toImplode = [];
            foreach ($var as $key => $value) {
                $toImplode[] = var_export($key, true).' => '.var_export_min($value, true);
            }
            $code = '['.implode(',', $toImplode).']';
            if ($return) return $code;
            else echo $code;
        } else {
            // null var_export 会变成 NULL
            if ($var === null) {
                if ($return) return 'null';
                else echo $var;
            }
            return var_export($var, $return);
        }
    }
}