<?php


namespace Mainto\DDDCore\Relation;

use Illuminate\Support\Str;

trait ModelRelationTrait {
    private array $relations = [];

    private function targetRepository (): string {
        $repositoryName = class_basename(__CLASS__).'Repository';
        $namespace = preg_replace("/(\\\\Model\\\\Entity\\\\.*)/m", '', __CLASS__);

        return $namespace.'\\Model\\Repository\\'.$repositoryName;
    }

    /**
     * @param string $localKey
     * @param string $targetKey
     * @param array $modelToMany   if not provider will auto set standard callable
     * @param array$modelsToMany if not provider will auto set standard callable
     * @return HasMany
     */
    public function hasMany (string $localKey, string $targetKey, array $modelToMany = [], array$modelsToMany = []): HasMany {
        if (!$modelToMany) {
           $modelsToMany = [$this->targetRepository(), sprintf("find%sBy%s", Str::studly(debug_backtrace(false, 2)[1]['function']), Str::studly($localKey))];
        }

        if (!$modelsToMany) {
           $modelsToMany = [$this->targetRepository(), sprintf("find%sBy%ss", Str::studly(debug_backtrace(false, 2)[1]['function']), Str::studly($localKey))];
        }

        return new HasMany(new HasManyDefaultProvider($localKey, $targetKey, $modelToMany,$modelsToMany));
    }

    /**
     *
     * @param string $localKey
     * @param string $targetKey
     * @param array $modelToOne   if not provider will auto set standard callable
     * @param array$modelsToOne if not provider will auto set standard callable
     * @return HasOne
     */
    public function hasOne (string $localKey, string $targetKey, array $modelToOne = [], array$modelsToOne = []): HasOne {
        if (!$modelToOne) {
            $modelToOne = [$this->targetRepository(), sprintf("find%sBy%s", Str::studly(debug_backtrace(false, 2)[1]['function']), Str::studly($localKey))];
        }

        if (!$modelsToOne) {
           $modelsToOne = [$this->targetRepository(), sprintf("find%sBy%ss", Str::plural(debug_backtrace(false, 2)[1]['function']), Str::studly($localKey))];
        }

        return new HasOne(new HasOneDefaultProvider($localKey, $targetKey, $modelToOne,$modelsToOne));
    }

    /**
     *
     * @param string $foreignKey
     * @param string $ownerKey
     * @param array $modelToOne   if not provider will auto set standard callable
     * @param array$modelsToOne if not provider will auto set standard callable
     * @return BelongsTo
     */
    public function belongsTo (string $foreignKey, string $ownerKey, array $modelToOne, array$modelsToOne): BelongsTo {
        if (!$modelToOne) {
            $modelToOne = [$this->targetRepository(), sprintf("find%sBy%s", Str::studly(debug_backtrace(false, 2)[1]['function']), Str::studly($ownerKey))];
        }

        if (!$modelsToOne) {
           $modelsToOne = [$this->targetRepository(), sprintf("find%sBy%ss", Str::plural(debug_backtrace(false, 2)[1]['function']), Str::studly($ownerKey))];
        }

        return new BelongsTo(new BelongsToDefaultProvider($foreignKey, $ownerKey, $modelToOne,$modelsToOne));
    }


    public function __get ($name) {
        if (array_key_exists($name, $this->relations)) {
            return $this->relations[$name];
        }

        if (method_exists($this, $name)) {
            $relation = $this->{$name}();
            if ($relation instanceof Relationship) {
                $keys = (array)$relation->getCurrentKey();
                $params = [];
                foreach ($keys as $key) {
                    $params[] = getter($this, $key);
                }
                if (!array_filter($params)) {
                    return $this->relations[$name] = null;
                }

                return $this->relations[$name] = $relation->modelResult(...$params);
            }
        }
    }

    public function setRelation ($relation, $value): self {
        $this->relations[$relation] = $value;

        return $this;
    }

    public function refresh ($relation): self {
        if (array_key_exists($relation, $this->relations)) {
            unset($this->relations[$relation]);
        }

        return $this;
    }
}
