<?php


namespace Mainto\DDDCore\Relation;


use Illuminate\Support\Str;
use Mainto\DDDCore\Exception\UnknownRelationshipException;
use Mainto\DDDCore\Exception\ModelKeyIsInvalid;
use Iterator;
use Mainto\DDDCore\Interfaces\Identity;

/**
 * Class ModelArray
 * @package Mainto\DDDCore
 * @template T
 */
class ModelArray implements Iterator {
    private array $loaded = [];

    /**
     * @var $models array<T|ModelRelation>
     */
    private array $models;

    /**
     * @param array $entities
     * @return ModelArray
     */
    public static function from(array $entities): ModelArray {
        return new self($entities);
    }

    /**
     * ModelArray constructor.
     * @param array<T> $entities
     */
    public function __construct (array $entities) {
        $this->models = $entities;
    }

    /**
     * @param string $relation
     * @param Relationship $relationship
     */
    private function resolveRelationship(string $relation, Relationship $relationship) {
        $oneResult = false;
        if ($relationship instanceof HasOne || $relationship instanceof BelongsTo) {
            $oneResult = true;
        }

        $modelMap = $localUniqueKeys = [];
        foreach ($this->models as $model) {
            $localKey = $this->getter($model, $relationship->getCurrentKey());
            if ($localKey instanceof Identity) {
                $localSerializeValue = $localKey->value();
            } elseif (is_string($localKey) || is_int($localKey)) {
                $localSerializeValue = $localKey;
            } elseif (is_null($localKey)) {
                goto init;
            } else {
                throw new ModelKeyIsInvalid();
            }

            if (!isset($modelMap[$localSerializeValue])) {
                $localUniqueKeys[] = $localKey;
            }

            $modelMap[$localSerializeValue][] = $model;
init:
            $model->setRelation($relation, $oneResult ? null : []);
        }

        if ($localUniqueKeys) {
            $targetModels = $relationship->modelsResult($localUniqueKeys);

            $targetModelGroup = [];
            foreach ($targetModels as $targetModel) {
                $relationKey = $this->getter($targetModel, $relationship->getTargetKey());
                if ($relationKey instanceof Identity) {
                    $serializeValue = $relationKey->value();
                } elseif (is_string($relationKey) || is_int($relationKey)) {
                    $serializeValue = $relationKey;
                } else {
                    throw new ModelKeyIsInvalid();
                }

                $targetModelGroup[$serializeValue][] = $targetModel;
            }

            foreach ($targetModelGroup as $serializeValue => $relationModels) {
                if (array_key_exists($serializeValue, $modelMap)) {
                    $models = $modelMap[$serializeValue];
                    foreach ($models as $model) {
                        if ($oneResult) {
                            $model->setRelation($relation, array_first($relationModels));
                        } else {
                            $model->setRelation($relation, $relationModels);
                        }
                    }
                }
            }
        }

        $this->loaded[$relation] = true;
    }

    public function load (...$relation): self {
        foreach ($relation as $oneRelation) {
            if (!array_key_exists($oneRelation, $this->loaded)) {
                $first = array_first($this->models);
                if (method_exists($first, $oneRelation)) {
                    $relationship = $first->{$oneRelation}();
                    if ($relationship instanceof Relationship) {
                        $this->resolveRelationship($oneRelation, $relationship);
                    } else {
                        throw new UnknownRelationshipException();
                    }
                } else {
                    throw new UnknownRelationshipException();
                }
            }
        }

        return $this;
    }

    /**
     * @param $relation
     * @return ModelArray
     */
    public function refresh ($relation): self {
        if (array_key_exists($relation, $this->models)) {
            foreach ($this->models as $item) {
                $item->refresh($relation);
            }

            unset($this->models[$relation]);
        }

        return $this;
    }

    /**
     * @return T
     */
    public function current () {
        return current($this->models);
    }

    public function next () {
        next($this->models);
    }

    public function key () {
        return key($this->models);
    }

    public function valid (): bool {
        return key($this->models) !== null;
    }

    public function rewind () {
        reset($this->models);
    }

    public function filter (callable $callback): array {
        return array_filter($this->models, $callback);
    }

    public function map (callable $callback): array {
        $map = [];
        foreach ($this->models as $key => $value) {
            $map[] = $callback($value, $key);
        }

        return $map;
    }

    public function pluck($value, $key = null): array {
        $res = [];

        foreach ($this->models as $model) {
            if ($key) {
                $res[$this->getter($model, $key)] = $this->getter($model, $value);
            } else {
                $res[] = $this->getter($model, $value);
            }
        }

        return $res;
    }

    public function each(callable $callback) {
        foreach ($this->models as $key => $value) {
            if ($callback($value, $key) === false) {
                break;
            }
        }
    }

    private function getter(object $object, string $property) {
        $getMethod = 'get'.Str::studly($property);
        if (method_exists($object, $getMethod)) {
            $value = $object->{$getMethod}();
        } else {
            $value = $object->{$property};
        }

        return $value;
    }
}
