<?php


namespace Mainto\DDDCore\Relation;

use Iterator;
use Mainto\DDDCore\Exception\RelationshipNotLoadedException;

trait ModelRelationTrait {
    private array $relations = [];

    public function with (...$relationGroup): self {
        foreach ($relationGroup as $relations) {
            $relationArr = explode('.', $relations);
            $current = $this;
            foreach ($relationArr as $relation) {
                if (is_array($current)) {
                    $current = (new ModelArray($current))->load($relation)->pluck($relation);
                } else if ($current instanceof ModelArray) {
                    $current = $current->load($relation)->pluck($relation);
                } else {
                    $current = $current->{$relation};
                }
            }
        }

        return $this;
    }

    /**
     * @param string $localKey
     * @param string $targetKey
     * @param array $modelToMany
     * @param array $modelsToMany
     * @return HasMany
     */
    public function hasMany (string $localKey, string $targetKey, array $modelToMany = [], array $modelsToMany = []): HasMany {
        return new HasMany(new HasManyDefaultProvider($localKey, $targetKey, $modelToMany, $modelsToMany));
    }

    /**
     *
     * @param string $localKey
     * @param string $targetKey
     * @param array $modelToOne
     * @param array $modelsToOne
     * @return HasOne
     */
    public function hasOne (string $localKey, string $targetKey, array $modelToOne = [], array $modelsToOne = []): HasOne {
        return new HasOne(new HasOneDefaultProvider($localKey, $targetKey, $modelToOne, $modelsToOne));
    }

    /**
     *
     * @param string $foreignKey
     * @param string $ownerKey
     * @param array $modelToOne
     * @param array $modelsToOne
     * @return BelongsTo
     */
    public function belongsTo (string $foreignKey, string $ownerKey, array $modelToOne, array $modelsToOne): BelongsTo {
        return new BelongsTo(new BelongsToDefaultProvider($foreignKey, $ownerKey, $modelToOne, $modelsToOne));
    }


    public function __get ($name) {
        if ($this->loaded($name)) {
            return $this->relations[$name];
        }

        if (method_exists($this, $name)) {
            return $this->resolveRelationship($name);
        }
    }

    public function setRelation ($relation, $value): self {
        $this->relations[$relation] = $value;

        return $this;
    }

    public function loaded (string $relation): bool {
        return array_key_exists($relation, $this->relations);
    }

    public function getRelation (string $relation, $autoLoad = true) {
        if ($this->loaded($relation)) {
            return $this->relations[$relation];
        }

        if ($autoLoad) {
            return $this->{$relation};
        }

        throw new RelationshipNotLoadedException();
    }

    public function refresh ($relation): self {
        if (array_key_exists($relation, $this->relations)) {
            unset($this->relations[$relation]);
        }

        return $this;
    }

    /**
     * @param string $relationName
     * @return iterable|object|void|null
     */
    public function resolveRelationship (string $relationName) {
        $relation = $this->{$relationName}();
        if ($relation instanceof Relationship) {
            $value = getter($this, $relation->getCurrentKey());
            if (is_null($value)) {
                return $this->relations[$relationName] = null;
            }
            return $this->relations[$relationName] = $relation->modelResult($value);
        }
    }
}
