<?php


namespace Mainto\DDDCore\Relation;

use Illuminate\Support\Str;

trait ModelRelationTrait {
    private array $relations = [];

    private function targetRepository (): string {
        $repositoryName = class_basename(__CLASS__).'Repository';
        $namespace = preg_replace("/(\\\\Model\\\\Entity\\\\.*)/m", '', __CLASS__);

        return $namespace.'\\Model\\Repository\\'.$repositoryName;
    }

    /**
     * @param string $localKey
     * @param string $targetKey
     * @param array $entityToMany   if not provider will auto set standard callable
     * @param array $entitiesToMany if not provider will auto set standard callable
     * @return HasMany
     */
    public function hasMany (string $localKey, string $targetKey, array $entityToMany = [], array $entitiesToMany = []): HasMany {
        if (!$entityToMany) {
            $entitiesToMany = [$this->targetRepository(), sprintf("find%sBy%s", Str::studly(debug_backtrace(false, 2)[1]['function']), Str::studly($localKey))];
        }

        if (!$entitiesToMany) {
            $entitiesToMany = [$this->targetRepository(), sprintf("find%sBy%ss", Str::studly(debug_backtrace(false, 2)[1]['function']), Str::studly($localKey))];
        }

        return new HasMany(new HasManyDefaultProvider($localKey, $targetKey, $entityToMany, $entitiesToMany));
    }

    /**
     *
     * @param string $localKey
     * @param string $targetKey
     * @param array $entityToOne   if not provider will auto set standard callable
     * @param array $entitiesToOne if not provider will auto set standard callable
     * @return HasOne
     */
    public function hasOne (string $localKey, string $targetKey, array $entityToOne = [], array $entitiesToOne = []): HasOne {
        if (!$entityToOne) {
            $entityToOne = [$this->targetRepository(), sprintf("find%sBy%s", Str::studly(debug_backtrace(false, 2)[1]['function']), Str::studly($localKey))];
        }

        if (!$entitiesToOne) {
            $entitiesToOne = [$this->targetRepository(), sprintf("find%sBy%ss", Str::plural(debug_backtrace(false, 2)[1]['function']), Str::studly($localKey))];
        }

        return new HasOne(new HasOneDefaultProvider($localKey, $targetKey, $entityToOne, $entitiesToOne));
    }

    /**
     *
     * @param string $foreignKey
     * @param string $ownerKey
     * @param array $entityToOne   if not provider will auto set standard callable
     * @param array $entitiesToOne if not provider will auto set standard callable
     * @return BelongsTo
     */
    public function belongsTo (string $foreignKey, string $ownerKey, array $entityToOne, array $entitiesToOne): BelongsTo {
        if (!$entityToOne) {
            $entityToOne = [$this->targetRepository(), sprintf("find%sBy%s", Str::studly(debug_backtrace(false, 2)[1]['function']), Str::studly($ownerKey))];
        }

        if (!$entitiesToOne) {
            $entitiesToOne = [$this->targetRepository(), sprintf("find%sBy%ss", Str::plural(debug_backtrace(false, 2)[1]['function']), Str::studly($ownerKey))];
        }

        return new BelongsTo(new BelongsToDefaultProvider($foreignKey, $ownerKey, $entityToOne, $entitiesToOne));
    }


    public function __get ($name) {
        if (array_key_exists($name, $this->relations)) {
            return $this->relations[$name];
        }

        if (method_exists($this, $name)) {
            $relation = $this->{$name}();
            if ($relation instanceof Relationship) {
                $value = $this->{$relation->getCurrentKey()};
                if (is_null($value)) {
                    return $this->relations[$name] = null;
                }
                return $this->relations[$name] = $relation->modelResult($this->{$relation->getCurrentKey()});
            }
        }
    }

    public function setRelation ($relation, $value): self {
        $this->relations[$relation] = $value;

        return $this;
    }

    public function refresh ($relation): self {
        if (array_key_exists($relation, $this->relations)) {
            unset($this->relations[$relation]);
        }

        return $this;
    }
}
