<?php


namespace Mainto\DDDCore\Relation;


use Illuminate\Support\Arr;
use Mainto\DDDCore\Exception\EntityPropertyNotExistsException;
use Mainto\DDDCore\Exception\UnknownRelationshipException;
use Mainto\DDDCore\Exception\EntityKeyIsInvalid;
use Iterator;
use Mainto\DDDCore\Interfaces\ValueId;

/**
 * Class EntityArray
 * @package Mainto\DDDCore
 * @template T
 */
class EntityArray implements Iterator {
    private array $loaded = [];

    /**
     * @var $entities array<T|EntityRelation>
     */
    private array $entities;

    /**
     * @param array $entities
     * @return EntityArray
     */
    public static function from(array $entities): EntityArray {
        return new self($entities);
    }

    /**
     * EntityArray constructor.
     * @param array<T> $entities
     */
    public function __construct (array $entities) {
        $this->entities = $entities;
    }

    /**
     * @param string $relation
     * @param Relationship $relationship
     */
    private function resolveRelationship(string $relation, Relationship $relationship) {
        $oneResult = false;
        if ($relationship instanceof HasOne || $relationship instanceof BelongsTo) {
            $oneResult = true;
        }

        $entityMap = $localUniqueKeys = [];
        foreach ($this->entities as $entity) {
            $localKey = $entity->{$relationship->getCurrentKey()};
            if ($localKey instanceof ValueId) {
                $localSerializeValue = $localKey->id();
            } elseif (is_string($localKey) || is_int($localKey)) {
                $localSerializeValue = $localKey;
            } else {
                throw new EntityKeyIsInvalid();
            }

            if (!isset($entityMap[$localSerializeValue])) {
                $localUniqueKeys[] = $localKey;
            }

            $entityMap[$localSerializeValue][] = $entity;
            $entity->setRelation($relation, $oneResult ? null : []);
        }

        if ($localUniqueKeys) {
            $targetEntities = $relationship->entitiesResult($localUniqueKeys);

            $targetEntityGroup = [];
            foreach ($targetEntities as $targetEntity) {
                $relationKey = $targetEntity->{$relationship->getTargetKey()};
                if ($relationKey instanceof ValueId) {
                    $serializeValue = $relationKey->id();
                } elseif (is_string($relationKey) || is_int($relationKey)) {
                    $serializeValue = $relationKey;
                } else {
                    throw new EntityKeyIsInvalid();
                }

                $targetEntityGroup[$serializeValue][] = $targetEntity;
            }

            foreach ($targetEntityGroup as $serializeValue => $relationEntities) {
                if (array_key_exists($serializeValue, $entityMap)) {
                    $entities = $entityMap[$serializeValue];
                    foreach ($entities as $entity) {
                        if ($oneResult) {
                            $entity->setRelation($relation, array_first($relationEntities));
                        } else {
                            $entity->setRelation($relation, $relationEntities);
                        }
                    }
                }
            }
        }

        $this->loaded[$relation] = true;
    }

    public function load (...$relation): self {
        foreach ($relation as $oneRelation) {
            if (!array_key_exists($oneRelation, $this->loaded)) {
                $first = array_first($this->entities);
                if (method_exists($first, $oneRelation)) {
                    $relationship = $first->{$oneRelation}();
                    if ($relationship instanceof Relationship) {
                        $this->resolveRelationship($oneRelation, $relationship);
                    } else {
                        throw new UnknownRelationshipException();
                    }
                } else {
                    throw new UnknownRelationshipException();
                }
            }
        }

        return $this;
    }

    /**
     * @param $relation
     * @return EntityArray
     */
    public function refresh ($relation): self {
        if (array_key_exists($relation, $this->entities)) {
            foreach ($this->entities as $item) {
                $item->refresh($relation);
            }

            unset($this->entities[$relation]);
        }

        return $this;
    }

    /**
     * @return T
     */
    public function current () {
        return current($this->entities);
    }

    public function next () {
        next($this->entities);
    }

    public function key () {
        return key($this->entities);
    }

    public function valid (): bool {
        return key($this->entities) !== null;
    }

    public function rewind () {
        reset($this->entities);
    }

    public function filter (callable $callback): array {
        return array_filter($this->entities, $callback);
    }

    public function map (callable $callback): array {
        $map = [];
        foreach ($this->entities as $key => $value) {
            $map[] = $callback($value, $key);
        }

        return $map;
    }

    public function pluck($value, $key = null): array {
        $res = [];

        foreach ($this->entities as $entity) {
            if ($key) {
                $res[$entity->{$key}] = $entity->{$value};
            } else {
                $res[] = $entity->{$value};
            }
        }

        return $res;
    }

    public function each(callable $callback) {
        foreach ($this->entities as $key => $value) {
            if ($callback($value, $key) === false) {
                break;
            }
        }
    }
}
