<?php


namespace Mainto\DDDCore\Relation;

use Illuminate\Support\Str;

trait EntityRelationTrait {
    private array $relations = [];

    private function targetRepository(): string {
        $repositoryName = class_basename(__CLASS__) . 'Repository';
        $namespace = preg_replace("/(\\\\Model\\\\Entity\\\\.*)/m", '', __CLASS__);

        return $namespace .'\\Model\\Repository\\'.$repositoryName;
    }

    /**
     * @param string $localKey
     * @param string $targetKey
     * @param array $entityToMany if not provider will auto set standard callable
     * @param array $entitiesToMany if not provider will auto set standard callable
     * @return HasMany
     */
    public function hasMany(string $localKey, string $targetKey, array $entityToMany = [], array $entitiesToMany = []): HasMany {
        if (!$entityToMany) {
            $entitiesToMany = [$this->targetRepository(), sprintf("find%sBy%s", Str::studly(debug_backtrace(false, 2)[1]['function']), Str::studly($localKey))];
        }

        if (!$entitiesToMany) {
            $entitiesToMany = [$this->targetRepository(), sprintf("find%sBy%ss", Str::studly(debug_backtrace(false, 2)[1]['function']), Str::studly($localKey))];
        }

        return new HasMany(new HasManyDefaultProvider($localKey, $targetKey, $entityToMany, $entitiesToMany));
    }

    /**
     *
     * @param string $localKey
     * @param string $targetKey
     * @param array $entityToOne if not provider will auto set standard callable
     * @param array $entitiesToOne if not provider will auto set standard callable
     * @return HasOne
     */
    public function hasOne(string $localKey, string $targetKey, array $entityToOne = [], array $entitiesToOne = []): HasOne {
        if (!$entityToOne) {
            $entityToOne = [$this->targetRepository(), sprintf("find%sBy%s", Str::studly(debug_backtrace(false, 2)[1]['function']), Str::studly($localKey))];
        }

        if (!$entitiesToOne) {
            $entitiesToOne = [$this->targetRepository(), sprintf("find%sBy%ss", Str::studly(debug_backtrace(false, 2)[1]['function']), Str::studly($localKey))];
        }

        return new HasOne(new HasOneDefaultProvider($localKey, $targetKey, $entityToOne, $entitiesToOne));
    }

    public function __get ($name) {
        if (array_key_exists($name, $this->relations)) {
            return $this->relations[$name];
        }

        if (method_exists($this, $name)) {
            $relation = $this->{$name}();
            if ($relation instanceof HasMany) {
                return $this->relations[$name] = $relation->entityResult($this->{$relation->getLocalKey()});
            } elseif ($relation instanceof HasOne) {
                return $this->relations[$name] = $relation->entityResult($this->{$relation->getLocalKey()});
            }
        }
    }

    public function setRelation($relation, $value): self {
        $this->relations[$relation] = $value;

        return $this;
    }

    public function refresh($relation): self {
        if (array_key_exists($relation, $this->relations)) {
            unset($this->relations[$relation]);
        }

        return $this;
    }
}
